unit PegasusControl;

{$undef USE_CODESITE}


interface

uses
  PegasusProtocol,
  classes, extctrls, messages, windows;

type
  // helper types
  // used to notify user object of a change in a property
  TWhichProperty =
    (
    wpHeartbeat, wpTunerDone, wpStartTx, wpStartRx,
    wpRxMeter, wpTxMeter, wpEncoder, wpPodKeypress,
    wpDSPSignalLevel, wpAnalogAGCLevel, wpForwardPower, wpReflectedPower,
    wpBadSyntax,

    // properties
    wpComPort,
    wpCWPitch,
    wpRxMode, wpRxFrequency, wpRxFilter, wpRxPBT, wpRxPBTLevel, wpRIT, wpRITLevel,
    wpTxMode, wpTxFrequency, wpTxFilter, wpTxPBT, wpTxPBTLevel, wpXIT, wpXITLevel,
    wpNR, wpAutoNotch, wpKeyerEnable,
    wpKeyerSpeed, wpKeyerWeight,wpPaddleReverse, wpAuxGain, wpMicGain,
    wpAuxInput, wpLineLevel, wpMute, wpMuteOnExit, wpAFGain, wpSidetone,
    wpSidetoneLevel, wpSpot, wpSpotLevel, wpMonitor, wpMonitorLevel,
    wpMonitorBoost, wpRFGain, wpRFAtten, wpAGC, wpNB, wpNBLevel, wpSquelch,
    wpSquelchLevel, wpRFOut, wpQSK, wpTxHang, wpTLoop, wpTxEnable, wpKeyDown,
    wpTimeout, wpWidebandTx, wpSpeechProcessor, wpSpeechProcessorLevel,
    wpAMCarrier, wpVOX, wpVOXGain, wpVOXAnti, wpVOXHang, wpTunerRFOut, wpSWRRFOut
    );

type
  TVFORec = record
    bCWPitch     : word;

    bRxMode      : byte;
    bRxFrequency : longint;
    bRxFilter    : byte;
    bRxPBT       : Boolean;
    bRxPBTLevel  : smallint;
    bRIT         : Boolean;
    bRITLevel    : smallint;

    bTxMode      : byte;
    bTxFrequency : longint;
    bTxFilter    : byte;
    bTxPBT       : Boolean;
    bTxPBTLevel  : smallint;
    bXIT         : Boolean;
    bXITLevel    : smallint;
  end;

  TPegasusControlRec = record
    // group NR / AutoNotch
    bNR : Boolean;
    bAutoNotch : Boolean;

    bKeyerEnable : Boolean;

    // group input source / input gain
    bAuxGain : byte;
    bMicGain : byte;
    bAuxInput : Boolean;

    // group keyer
    bKeyerSpeed : byte;
    bKeyerWeight : byte;
    bPaddleReverse : Boolean;

    bLineLevel     : byte;
    bMute          : Boolean;
    bMuteOnExit    : Boolean;
    bAFGain        : byte;
    bSidetone      : Boolean;
    bSidetoneLevel : byte;
    bSpot          : Boolean;
    bSpotLevel     : byte;
    bMonitor       : Boolean;
    bMonitorLevel  : byte;
    bMonitorBoost  : byte;

    bRFGain        : byte;
    bRFAtten       : Boolean;
    bAGC           : byte;
    bNB            : Boolean;
    bNBLevel       : byte;
    bSquelch       : Boolean;
    bSquelchLevel  : byte;

    bRFOut                : byte;
    bQSK                  : byte;
    bTxHang               : byte;
    bTLoop                : Boolean;
    bTxEnable             : Boolean;
    bKeyDown              : Boolean;
    bTimeout              : Boolean;
    bWidebandTx           : Boolean;
    bSpeechProcessor      : Boolean;
    bSpeechProcessorLevel : byte;
    bAMCarrier            : byte;

    bVOX     : Boolean;
    bVOXGain : byte;
    bVOXAnti : byte;
    bVOXHang : byte;

    bTunerRFOut : byte;

    bSWRRFOut : byte;
  end;

  TTunerStep = ( tunerNone, tunerRFOn, tunerCmd, tunerWait, tunerSWR );


type
  TmePegasusControl = class;

  TPegCNotifyEvent = procedure( const Sender : TmePegasusControl ) of object;

  TmePegasusControl = class( TComponent )
  private
    fPegasusProtocol : TPegasusProtocol;
    fWindowHandle : HWND;

    fComPort : byte;

    fVFORec : TVFORec;
    fControlRec : TPegasusControlRec;

    // track multi property groups

    // - InputSource / Gain
    fInUpdateInputSourceGain : Boolean;
    fHoldAuxGain             : byte;
    fHoldMicGain             : byte;
    fHoldAuxInput            : Boolean;

    // - noise reduction / auto notch
    fInUpdateNRAutoNotch : Boolean;
    fHoldNR              : Boolean;
    fHoldAutoNotch       : Boolean;

    // - keyer
    fInUpdateKeyer     : Boolean;
    fHoldKeyerSpeed    : byte;
    fHoldKeyerWeight   : byte;
    fHoldPaddleReverse : Boolean;

    // - VFO
    fInUpdateVFO     : Boolean;
    fHoldVFO         : TVFORec;

    // misc
    fDSPVersion  : integer;
    fDoNotify    : Boolean;
    fForceUpdate : Boolean;

    // heartbeat
    fHeartbeat   : TTimer;
    fLastStateTx : Boolean;

    fOnChangeArray : array[TWhichProperty] of TPegCNotifyEvent;

    // power
    fLastReflected : byte;
    fLastForward   : byte;

    // Tuner
    fTunerActive  : Boolean;
    fTunerCommand : char;

    fTunerStep    : TTunerStep;
    fTunerCtr     : smallint;
    fTunerDupCtr  : smallint;
    fTunerFwd     : byte;
    fTunerRef     : byte;
    fTunerOk      : Boolean;

    fTunerHoldRxMode : byte;
    fTunerHoldTxMode : byte;

    // pod
    fLastEncoder     : smallint;
    fLastEncoderKey  : char;
    fLastPodKeypress : char;

    // S-Meter
    fSMeterFrequency : longint;
    fLastSMeter      : word;

    // misc messages
    fLastDSPSignalLevel : word;
    fLastAnalogAGCLevel : byte;

    fInUpdateNotify : Boolean;

    fNeedRestart : Boolean;

    // Heartbeat events
    procedure DoHeartbeat( sender : TObject );
    procedure DoTunerHeartbeat;

    // TPegausProtocol event handlers

    // support
    class function CalcOnOffInteger( const aOn : Boolean; const aValue : integer ) : integer;
    class function CalcAdjFrequency( const aOn : Boolean; const aFrequency, aAdjust : longint ) : longint;
    class procedure SetOnOffByte( const aOn : Boolean; const aLevel : byte; const aWhich : TPegPSetByte );
    class function AdjustFilter( const aValue : byte ) : byte;

    procedure RxTxModesInt( const aRxMode, aTxMode : byte );

    procedure UpdateRxParams;
    procedure UpdateTxParams;
    procedure KeyerParams;
    procedure SetupRadio;
    procedure Tuner( const aWhich : char );

    // property access methods
    procedure SetComPort( const value : byte );  // 0 - close; 1 .. open
    procedure SetAFGain(const Value: byte);
    procedure SetAGC(const Value: byte);
    procedure SetAMCarrier(const Value: byte);
    procedure SetAutoNotch(const Value: Boolean);
    procedure SetAuxGain(const Value: byte);
    procedure SetAuxInput(const Value: Boolean);
    procedure SetCWPitch( const Value: word );
    procedure SetKeyDown(const Value: Boolean);
    procedure SetKeyerEnable(const Value: Boolean);
    procedure SetLineLevel(const Value: byte);
    procedure SetMicGain(const Value: byte);
    procedure SetMonitorLevel(const Value: byte);
    procedure SetMonitorBoost( const value : byte );
    procedure SetMute(const Value: Boolean);
    procedure SetNBLevel(const Value: byte);
    procedure SetNR(const Value: Boolean);
    procedure SetQSK(const Value: byte);
    procedure SetRFAtten(const Value: Boolean);
    procedure SetRFGain(const Value: byte);
    procedure SetRFOut(const Value: byte);
    procedure SetRxFilter(const Value: byte);
    procedure SetRxFrequency(const Value: longint);
    procedure SetRxMode(const Value: byte);
    procedure SetRxPBT( const value : Boolean );
    procedure SetRxPBTLevel(const Value: smallint);
    procedure SetRIT( const value : Boolean );
    procedure SetRITLevel( const value : smallint );
    procedure SetSidetoneLevel(const Value: byte);
    procedure SetSpeechProcessorLevel(const Value: byte);
    procedure SetSpotLevel(const Value: byte);
    procedure SetSquelchLevel( const value : Byte );
    procedure SetSquelch(const Value: Boolean);
    procedure SetTimeout(const Value: Boolean);
    procedure SetTLoop(const Value: Boolean);
    procedure SetTxEnable(const Value: Boolean);
    procedure SetTxFilter(const Value: byte);
    procedure SetTxFrequency(const Value: longint);
    procedure SetTxHang(const Value: byte);
    procedure SetTxMode(const Value: byte);
    procedure SetTxPBT( const value : Boolean );
    procedure SetTxPBTLevel(const Value: smallint );
    procedure SetXIT( const value : Boolean );
    procedure SetXITLevel( const value : smallint );
    procedure SetVOX(const Value: Boolean);
    procedure SetVOXAnti(const Value: byte);
    procedure SetVOXGain(const Value: byte);
    procedure SetVOXHang(const Value: byte);
    procedure SetWidebandTx(const Value: Boolean);

    procedure SetNB( const value : Boolean );
    procedure SetSpeechProcessor( const value : Boolean );
    procedure SetMonitor(const Value: Boolean);
    procedure SetSidetone(const Value: Boolean);
    procedure SetSpot(const Value: Boolean);

    procedure SetKeyerSpeed( const value : byte );
    procedure SetKeyerWeight( const value : byte );
    procedure SetPaddleReverse( const value : Boolean );

    procedure SetTunerRFOut( const value : byte );
    procedure SetSWRRFOut( const value : byte );

    procedure SetCurrentVFO( const value : TVFORec );
    procedure SetCurrentParams( const value : TPegasusControlRec );



    function GetLastSWR : double;
    procedure DoPropertyChange( const aWhichProperty : TWhichProperty );
    procedure CheckMonitorBoost;

    function GetOnChange( const aWhich : TWhichProperty ) : TPegCNotifyEvent;
    procedure SetOnChange( const aWhich : TWhichProperty; const aValue : TPegCNotifyEvent );

    procedure LoadDefaults;

    procedure WndProc( var aMsg : TMessage );

    procedure DoDSPStart;
    procedure DoRadioStart;
    procedure DoEncoder( const aMovement, aKeyPressed : integer );
    procedure DoForward( const aForward : integer );
    procedure DoReflected( const aReflected : integer );
    procedure DoRxMeter( const aSMeter : integer );
    procedure DoTxMeter( const aForward, aReflected : integer );
    procedure DoPodKey( const aKeyPressed : integer );
    procedure DoVersion( const aVersion : integer );
    procedure DoDigLevel( const aLevel : integer );
    procedure DoAnaLevel( const aLevel : integer );
    procedure DoBadSyntax;

// TmePegasusControl
  public
    constructor create( aOwner : TComponent ); override;
    destructor destroy; override;

    // Tuner Commands
    procedure PT11Bypass;
    procedure PT11AutoTune;
    procedure PT11CapUp;
    procedure PT11CapDown;
    procedure PT11IndUp;
    procedure PT11IndDown;
    procedure PT11HiZ;
    procedure PT11LoZ;
    procedure AT11Tune;
    procedure StopTune;
    procedure MeasureSWR;   // works with or w/o tuner

    property TunerOK : Boolean read fTunerOK;

    // Status - r/o properties
    property DSPVersion      : integer  read fDSPVersion;
    property LastForward     : byte     read fLastForward;
    property LastReflected   : byte     read fLastReflected;
    property LastSWR         : double   read GetLastSWR;
    property LastEncoder     : smallint read fLastEncoder;
    property LastEncoderKey  : char     read fLastEncoderKey;
    property LastPodKeypress : char     read fLastPodKeypress;

    // group property set methods
    procedure BeginUpdateVFO;
    procedure EndUpdateVFO;

    procedure BeginUpdateNRAutoNotch;
    procedure EndUpdateNRAutoNotch;

    procedure BeginUpdateInputSourceGain;
    procedure EndUpdateInputSourceGain;

    procedure BeginUpdateKeyer;
    procedure EndUpdateKeyer;

    // misc messages
    property LastDSPSignalLevel : word read fLastDSPSignalLevel;
    property LastAnalogAGCLevel : byte read fLastAnalogAGCLevel;

    // S-Meter
    property SMeterFrequency : longint read fSMeterFrequency;
    property LastSMeter      : word    read fLastSMeter;

    property InUpdateNotify : Boolean read fInUpdateNotify;

    // ***** properties *****
    property ComPort     : byte     read fComPort write SetComPort;
    // group VFO
    property CWPitch     : word     read fVFORec.bCWPitch     write SetCWPitch;
    property RxMode      : byte     read fVFORec.bRxMode      write SetRxMode;
    property RxFrequency : longint  read fVFORec.bRxFrequency write SetRxFrequency;
    property RxFilter    : byte     read fVFORec.bRxFilter    write SetRxFilter;
    property RxPBT       : Boolean  read fVFORec.bRxPBT       write SetRxPBT;
    property RxPBTLevel  : smallint read fVFORec.bRxPBTLevel  write SetRxPBTLevel;
    property RIT         : Boolean  read fVFORec.bRIT         write SetRIT;
    property RITLevel    : smallint read fVFORec.bRITLevel    write SetRITLevel;
    property TxMode      : byte     read fVFORec.bTxMode      write SetTxMode;
    property TxFrequency : longint  read fVFORec.bTxFrequency write SetTxFrequency;
    property TxFilter    : byte     read fVFORec.bTxFilter    write SetTxFilter;
    property TxPBT       : Boolean  read fVFORec.bTxPBT       write SetTxPBT;
    property TxPBTLevel  : smallint read fVFORec.bTxPBTLevel  write SetTxPBTLevel;
    property XIT         : Boolean  read fVFORec.bXIT         write SetXIT;
    property XITLevel    : smallint read fVFORec.bXITLevel    write SetXITLevel;

    // group NR / AutoNotch
    property NR        : Boolean read fControlRec.bNR        write SetNR;
    property AutoNotch : Boolean read fControlRec.bAutoNotch write SetAutoNotch;

    property KeyerEnable : Boolean read fControlRec.bKeyerEnable write SetKeyerEnable;

    // group input source / input gain
    property AuxGain  : byte    read fControlRec.bAuxGain  write SetAuxGain;
    property MicGain  : byte    read fControlRec.bMicGain  write SetMicGain;
    property AuxInput : Boolean read fControlRec.bAuxInput write SetAuxInput;

    // group keyer
    property KeyerSpeed    : byte    read fControlRec.bKeyerSpeed    write SetKeyerSpeed;
    property KeyerWeight   : byte    read fControlRec.bKeyerWeight   write SetKeyerWeight;
    property PaddleReverse : Boolean read fControlRec.bPaddleReverse write SetPaddleReverse;

    property LineLevel     : byte    read fControlRec.bLineLevel     write SetLineLevel;
    property Mute          : Boolean read fControlRec.bMute          write SetMute;
    property MuteOnExit    : Boolean read fControlRec.bMuteOnExit    write fControlRec.bMuteOnExit;
    property AFGain        : byte    read fControlRec.bAFGain        write SetAFGain;
    property Sidetone      : Boolean read fControlRec.bSidetone      write SetSidetone;
    property SidetoneLevel : byte    read fControlRec.bSidetoneLevel write SetSidetoneLevel;
    property Spot          : Boolean read fControlRec.bSpot          write SetSpot;
    property SpotLevel     : byte    read fControlRec.bSpotLevel     write SetSpotLevel;
    property Monitor       : Boolean read fControlRec.bMonitor       write SetMonitor;
    property MonitorLevel  : byte    read fControlRec.bMonitorLevel  write SetMonitorLevel;
    property MonitorBoost  : byte    read fControlRec.bMonitorBoost  write SetMonitorBoost;

    property RFGain        : byte    read fControlRec.bRFGain       write SetRFGain;
    property RFAtten       : Boolean read fControlRec.bRFAtten      write SetRFAtten;
    property AGC           : byte    read fControlRec.bAGC          write SetAGC;
    property NB            : Boolean read fControlRec.bNB           write SetNB;
    property NBLevel       : byte    read fControlRec.bNBLevel      write SetNBLevel;
    property Squelch       : Boolean read fControlRec.bSquelch      write SetSquelch;
    property SquelchLevel  : byte    read fControlRec.bSquelchLevel write SetSquelchLevel;

    property RFOut                : byte    read fControlRec.bRFOut                write SetRFOut;
    property QSK                  : byte    read fControlRec.bQSK                  write SetQSK;
    property TxHang               : byte    read fControlRec.bTxHang               write SetTxHang;
    property TLoop                : Boolean read fControlRec.bTLoop                write SetTLoop;
    property TxEnable             : Boolean read fControlRec.bTxEnable             write SetTxEnable;
    property KeyDown              : Boolean read fControlRec.bKeyDown              write SetKeyDown;
    property Timeout              : Boolean read fControlRec.bTimeout              write SetTimeout;
    property WidebandTx           : Boolean read fControlRec.bWidebandTx           write SetWidebandTx;
    property SpeechProcessor      : Boolean read fControlRec.bSpeechProcessor      write SetSpeechProcessor;
    property SpeechProcessorLevel : byte    read fControlRec.bSpeechProcessorLevel write SetSpeechProcessorLevel;
    property AMCarrier            : byte    read fControlRec.bAMCarrier            write SetAMCarrier;

    property VOX     : Boolean read fControlRec.bVOX     write SetVOX;
    property VOXGain : byte    read fControlRec.bVOXGain write SetVOXGain;
    property VOXAnti : byte    read fControlRec.bVOXAnti write SetVOXAnti;
    property VOXHang : byte    read fControlRec.bVOXHang write SetVOXHang;

    property TunerRFOut : byte read fControlRec.bTunerRFOut write SetTunerRFOut;

    property SWRRFOut : byte read fControlRec.bSWRRFOut write SetSWRRFOut;


    // VFO
    property CurrentVFO : TVFORec read fVFORec write SetCurrentVFO;
    property CurrentParams : TPegasusControlRec read fControlRec write SetCurrentParams;

    // Query
    procedure QueryRawAuxStatus;
    procedure QueryForwardPower;
    procedure QueryReflectedPower;
    procedure QueryDSPSignalLevel;
    procedure QueryAnalogAGCLevel;

    procedure SetupKeyer(const aSpeed, aWeight: byte; const aReverse: Boolean);

published
    // client notification
    property OnHeartbeat            : TPegCNotifyEvent index wpHeartbeat            read GetOnChange write SetOnChange;
    property OnTunerDone            : TPegCNotifyEvent index wpTunerDone            read GetOnChange write SetOnChange;
    property OnStartTx              : TPegCNotifyEvent index wpStartTx              read GetOnChange write SetOnChange;
    property OnStartRx              : TPegCNotifyEvent index wpStartRx              read GetOnChange write SetOnChange;
    property OnRxMeter              : TPegCNotifyEvent index wpRxMeter              read GetOnChange write SetOnChange;
    property OnTxMeter              : TPegCNotifyEvent index wpTxMeter              read GetOnChange write SetOnChange;
    property OnEncoder              : TPegCNotifyEvent index wpEncoder              read GetOnChange write SetOnChange;
    property OnPodKeypress          : TPegCNotifyEvent index wpPodKeypress          read GetOnChange write SetOnChange;
    property OnDSPSignalLevel       : TPegCNotifyEvent index wpDSPSignalLevel       read GetOnChange write SetOnChange;
    property OnAnalogAGCLevel       : TPegCNotifyEvent index wpAnalogAGCLevel       read GetOnChange write SetOnChange;
    property OnForwardPower         : TPegCNotifyEvent index wpForwardPower         read GetOnChange write SetOnChange;
    property OnReflectedPower       : TPegCNotifyEvent index wpReflectedPower       read GetOnChange write SetOnChange;
    property OnBadSyntax            : TPegCNotifyEvent index wpReflectedPower       read GetOnChange write SetOnChange;
    // -----
    property OnComPort              : TPegCNotifyEvent index wpComPort              read GetOnChange write SetOnChange;
    property OnCWPitch              : TPegCNotifyEvent index wpCWPitch              read GetOnChange write SetOnChange;
    property OnRxMode               : TPegCNotifyEvent index wpRxMode               read GetOnChange write SetOnChange;
    property OnRxFrequency          : TPegCNotifyEvent index wpRxFrequency          read GetOnChange write SetOnChange;
    property OnRxFilter             : TPegCNotifyEvent index wpRxFilter             read GetOnChange write SetOnChange;
    property OnRxPBT                : TPegCNotifyEvent index wpRxPBT                read GetOnChange write SetOnChange;
    property OnRxPBTLevel           : TPegCNotifyEvent index wpRxPBTLevel           read GetOnChange write SetOnChange;
    property OnRIT                  : TPegCNotifyEvent index wpRIT                  read GetOnChange write SetOnChange;
    property OnRITLevel             : TPegCNotifyEvent index wpRITLevel             read GetOnChange write SetOnChange;
    property OnTxMode               : TPegCNotifyEvent index wpTxMode               read GetOnChange write SetOnChange;
    property OnTxFrequency          : TPegCNotifyEvent index wpTxFrequency          read GetOnChange write SetOnChange;
    property OnTxFilter             : TPegCNotifyEvent index wpTxFilter             read GetOnChange write SetOnChange;
    property OnTxPBT                : TPegCNotifyEvent index wpTxPBT                read GetOnChange write SetOnChange;
    property OnTxPBTLevel           : TPegCNotifyEvent index wpTxPBTLevel           read GetOnChange write SetOnChange;
    property OnXIT                  : TPegCNotifyEvent index wpXIT                  read GetOnChange write SetOnChange;
    property OnXITLevel             : TPegCNotifyEvent index wpXITLevel             read GetOnChange write SetOnChange;
    property OnNR                   : TPegCNotifyEvent index wpNR                   read GetOnChange write SetOnChange;
    property OnAutoNotch            : TPegCNotifyEvent index wpAutoNotch            read GetOnChange write SetOnChange;
    property OnKeyerEnable          : TPegCNotifyEvent index wpKeyerEnable          read GetOnChange write SetOnChange;
    property OnKeyerSpeed           : TPegCNotifyEvent index wpKeyerSpeed           read GetOnChange write SetOnChange;
    property OnKeyerWeight          : TPegCNotifyEvent index wpKeyerWeight          read GetOnChange write SetOnChange;
    property OnPaddleReverse        : TPegCNotifyEvent index wpPaddleReverse        read GetOnChange write SetOnChange;
    property OnAuxGain              : TPegCNotifyEvent index wpAuxGain              read GetOnChange write SetOnChange;
    property OnMicGain              : TPegCNotifyEvent index wpMicGain              read GetOnChange write SetOnChange;
    property OnAuxInput             : TPegCNotifyEvent index wpAuxInput             read GetOnChange write SetOnChange;
    property OnLineLevel            : TPegCNotifyEvent index wpLineLevel            read GetOnChange write SetOnChange;
    property OnMute                 : TPegCNotifyEvent index wpMute                 read GetOnChange write SetOnChange;
    property OnMuteOnExit           : TPegCNotifyEvent index wpMuteOnExit           read GetOnChange write SetOnChange;
    property OnAFGain               : TPegCNotifyEvent index wpAFGain               read GetOnChange write SetOnChange;
    property OnSidetone             : TPegCNotifyEvent index wpSidetone             read GetOnChange write SetOnChange;
    property OnSidetoneLevel        : TPegCNotifyEvent index wpSidetoneLevel        read GetOnChange write SetOnChange;
    property OnSpot                 : TPegCNotifyEvent index wpSpot                 read GetOnChange write SetOnChange;
    property OnSpotLevel            : TPegCNotifyEvent index wpSpotLevel            read GetOnChange write SetOnChange;
    property OnMonitor              : TPegCNotifyEvent index wpMonitor              read GetOnChange write SetOnChange;
    property OnMonitorLevel         : TPegCNotifyEvent index wpMonitorLevel         read GetOnChange write SetOnChange;
    property OnMonitorBoost         : TPegCNotifyEvent index wpMonitorBoost         read GetOnChange write SetOnChange;
    property OnRFGain               : TPegCNotifyEvent index wpRFGain               read GetOnChange write SetOnChange;
    property OnRFAtten              : TPegCNotifyEvent index wpRFAtten              read GetOnChange write SetOnChange;
    property OnAGC                  : TPegCNotifyEvent index wpAGC                  read GetOnChange write SetOnChange;
    property OnNB                   : TPegCNotifyEvent index wpNB                   read GetOnChange write SetOnChange;
    property OnNBLevel              : TPegCNotifyEvent index wpNBLevel              read GetOnChange write SetOnChange;
    property OnSquelch              : TPegCNotifyEvent index wpSquelch              read GetOnChange write SetOnChange;
    property OnSquelchLevel         : TPegCNotifyEvent index wpSquelchLevel         read GetOnChange write SetOnChange;
    property OnRFOut                : TPegCNotifyEvent index wpRFOut                read GetOnChange write SetOnChange;
    property OnQSK                  : TPegCNotifyEvent index wpQSK                  read GetOnChange write SetOnChange;
    property OnTxHang               : TPegCNotifyEvent index wpTXHang               read GetOnChange write SetOnChange;
    property OnTLoop                : TPegCNotifyEvent index wpTLoop                read GetOnChange write SetOnChange;
    property OnTxEnable             : TPegCNotifyEvent index wpTxEnable             read GetOnChange write SetOnChange;
    property OnKeyDown              : TPegCNotifyEvent index wpKeyDown              read GetOnChange write SetOnChange;
    property OnTimeout              : TPegCNotifyEvent index wpTimeout              read GetOnChange write SetOnChange;
    property OnWidebandTx           : TPegCNotifyEvent index wpWidebandTx           read GetOnChange write SetOnChange;
    property OnSpeechProcessor      : TPegCNotifyEvent index wpSpeechProcessor      read GetOnChange write SetOnChange;
    property OnSpeechProcessorLevel : TPegCNotifyEvent index wpSpeechProcessorLevel read GetOnChange write SetOnChange;
    property OnAMCarrier            : TPegCNotifyEvent index wpAMCarrier            read GetOnChange write SetOnChange;
    property OnVOX                  : TPegCNotifyEvent index wpVOX                  read GetOnChange write SetOnChange;
    property OnVOXGain              : TPegCNotifyEvent index wpVOXGain              read GetOnChange write SetOnChange;
    property OnVOXAnti              : TPegCNotifyEvent index wpVOXAnti              read GetOnChange write SetOnChange;
    property OnVOXHang              : TPegCNotifyEvent index wpVOXHang              read GetOnChange write SetOnChange;
    property OnTunerRFOut           : TPegCNotifyEvent index wpTunerRFOut           read GetOnChange write SetOnChange;
    property OnSWRRFOut             : TPegCNotifyEvent index wpSWRRFOut             read GetOnChange write SetOnChange;
  end;


const
  // General purpose
  AGC_SLOW = 0; AGC_MED  = 1; AGC_FAST = 2;

  MODE_AM  = 0; MODE_USB = 1; MODE_LSB = 2; MODE_CW  = 3; MODE_FM  = 4;

  // note reodered from docs!
  FILTER_8000 =  0; FILTER_6000 =  1; FILTER_5700 =  2; FILTER_5400 =  3;
  FILTER_5100 =  4; FILTER_4800 =  5; FILTER_4500 =  6; FILTER_4200 =  7;
  FILTER_3900 =  8; FILTER_3600 =  9; FILTER_3300 = 10; FILTER_3000 = 11;
  FILTER_2850 = 12; FILTER_2700 = 13; FILTER_2550 = 14; FILTER_2400 = 15;
  FILTER_2250 = 16; FILTER_2100 = 17; FILTER_1950 = 18; FILTER_1800 = 19;
  FILTER_1650 = 20; FILTER_1500 = 21; FILTER_1350 = 22; FILTER_1200 = 23;
  FILTER_1050 = 24; FILTER_900  = 25; FILTER_750  = 26; FILTER_675  = 27;
  FILTER_600  = 28; FILTER_525  = 29; FILTER_450  = 30; FILTER_375  = 31;
  FILTER_330  = 32; FILTER_300  = 33;

  // note reodered from docs!
  FILTER_WIDTH : array[0 .. 33] of word =
    (
    8000, 6000, 5700, 5400, 5100, 4800, 4500, 4200, 3900, 3600, 3300, 3000, 2850,
    2700, 2550, 2400, 2250, 2100, 1950, 1800, 1650, 1500, 1350, 1200, 1050,  900,
     750,  675,  600,  525,  450,  375,  330,  300
    );

  MONITOR_BOOST_NONE = 0; MONITOR_BOOST_AF = 1; MONITOR_BOOST_BOTH = 2;

  HB_INTERVAL = 55;   //  heartbeat msec


  PT11_BYPASS = '0'; PT11_TUNE   = '1'; PT11_CAPUP  = '3'; PT11_CAPDN  = '4';
  PT11_INDUP  = '5'; PT11_INDDN  = '6'; PT11_HIZ    = '7'; PT11_LOZ    = '8';

  AT11_TUNE   = 'A';
  MEASURE_SWR = 'B';

  TUNER_MAX_FWD_DELTA = 1;
  TUNER_MAX_REF_DELTA = 1;

  TUNER_DUPCTR    =  1000 div HB_INTERVAL;
  TUNER_TIMEOUT   = 20000 div HB_INTERVAL;
  TUNER_SWR_PAUSE =   150 div HB_INTERVAL;


const
// defaults
  DEF_ComPort              = 0;      // disables
  DEF_CWPitch              = 600;
  DEF_RxMode               = MODE_USB;
  DEF_RxFrequency          = 14230000;
  DEF_RxFilter             = 17;
  DEF_RxPBT                = false;
  DEF_RxPBTLevel           = 0;
  DEF_RIT                  = false;
  DEF_RITLevel             = 0;
  DEF_TxMode               = MODE_USB;
  DEF_TxFrequency          = 14230000;
  DEF_TxFilter             = FILTER_2100;
  DEF_TxPBT                = false;
  DEF_TxPBTLevel           = 0;
  DEF_XIT                  = false;
  DEF_XITLevel             = 0;
  DEF_NR                   = false;
  DEF_AutoNotch            = false;
  DEF_KeyerEnable          = true;
  DEF_AuxGain              = 7;
  DEF_MicGain              = 7;
  DEF_AuxInput             = false;
  DEF_KeyerSpeed           = 15;
  DEF_KeyerWeight          = 100;
  DEF_PaddleReverse        = false;
  DEF_LineLevel            = 0; // max
  DEF_Mute                 = false;
  DEF_MuteOnExit           = true;
  DEF_AFGain               = 0;
  DEF_Sidetone             = false;
  DEF_SidetoneLevel        = 0;
  DEF_Spot                 = false;
  DEF_SpotLevel            = 0;
  DEF_Monitor              = false;
  DEF_MonitorLevel         = 0;
  DEF_MonitorBoost         = 0;
  DEF_RFGain               = 0;  // full
  DEF_RFAtten              = false;
  DEF_AGC                  = AGC_MED;
  DEF_NB                   = false;
  DEF_NBLevel              = 0;
  DEF_Squelch              = false;
  DEF_SquelchLevel         = 0;
  DEF_RFOut                = 0;
  DEF_QSK                  = 0;
  DEF_TxHang               = 0;
  DEF_TLoop                = false;
  DEF_TxEnable             = false;
  DEF_KeyDown              = false;
  DEF_Timeout              = true;
  DEF_WidebandTx           = false;
  DEF_SpeechProcessor      = false;
  DEF_SpeechProcessorLevel = 0;
  DEF_AMCarrier            = 0;
  DEF_VOX                  = false;
  DEF_VOXGain              = 0;
  DEF_VOXAnti              = 0;
  DEF_VOXHang              = 0;
  DEF_TunerRFOut           = 0;
  DEF_SWRRFOut             = 0;

  DEF_VFORec : TVFORec =
    (
    bCWPitch     : DEF_CWPitch;

    bRxMode      : DEF_RxMode;
    bRxFrequency : DEF_RxFrequency;
    bRxFilter    : DEF_RxFilter;
    bRxPBT       : DEF_RxPBT;
    bRxPBTLevel  : DEF_RxPBTLevel;
    bRIT         : DEF_RIT;
    bRITLevel    : DEF_RITLevel;

    bTxMode      : DEF_TxMode;
    bTxFrequency : DEF_TxFrequency;
    bTxFilter    : DEF_TxFilter;
    bTxPBT       : DEF_TxPBT;
    bTxPBTLevel  : DEF_TxPBTLevel;
    bXIT         : DEF_XIT;
    bXITLevel    : DEF_XITLevel
    );


  DEF_ControlRec : TPegasusControlRec =
    (
    bNR                   : DEF_NR;
    bAutoNotch            : DEF_AutoNotch;
    bKeyerEnable          : DEF_KeyerEnable;
    bAuxGain              : DEF_AuxGain;
    bMicGain              : DEF_MicGain;
    bAuxInput             : DEF_AuxInput;
    bKeyerSpeed           : DEF_KeyerSpeed;
    bKeyerWeight          : DEF_KeyerWeight;
    bPaddleReverse        : DEF_PaddleReverse;

    bLineLevel            : DEF_LineLevel;
    bMute                 : DEF_Mute;
    bMuteOnExit           : DEF_MuteOnExit;
    bAFGain               : DEF_AFGain;
    bSidetone             : DEF_Sidetone;
    bSidetoneLevel        : DEF_SidetoneLevel;
    bSpot                 : DEF_Spot;
    bSpotLevel            : DEF_SpotLevel;
    bMonitor              : DEF_Monitor;
    bMonitorLevel         : DEF_MonitorLevel;
    bMonitorBoost         : DEF_MonitorBoost;

    bRFGain               : DEF_RFGain;
    bRFAtten              : DEF_RFAtten;
    bAGC                  : DEF_AGC;
    bNB                   : DEF_NB;
    bNBLevel              : DEF_NBLevel;
    bSquelch              : DEF_Squelch;
    bSquelchLevel         : DEF_SquelchLevel;

    bRFOut                : DEF_RFOut;
    bQSK                  : DEF_QSK;
    bTxHang               : DEF_TxHang;
    bTLoop                : DEF_TLoop;
    bTxEnable             : DEF_TxEnable;
    bKeyDown              : DEF_KeyDown;
    bTimeout              : DEF_Timeout;
    bWidebandTx           : DEF_WidebandTx;
    bSpeechProcessor      : DEF_SpeechProcessor;
    bSpeechProcessorLevel : DEF_SpeechProcessorLevel;
    bAMCarrier            : DEF_AMCarrier;

    bVOX                  : DEF_VOX;
    bVOXGain              : DEF_VOXGain;
    bVOXAnti              : DEF_VOXAnti;
    bVOXHang              : DEF_VOXHang;

    bTunerRFOut           : DEF_TunerRFOut;

    bSWRRFOut             : DEF_SWRRFOut
    );

procedure Register;



implementation

{ TmePegasusControl }

uses
  {$ifdef USE_CODESITE}
  rzCSIntf,
  {$endif}
  sysutils;

// -----------------------------------------------------------------------------
// ----- AllocateHWnd stuff from FORMS.PAS -------------------------------------
// -----------------------------------------------------------------------------

const
  InstanceCount = 313;

type
  TAHWMethod = procedure(var Message : TMessage) of object;  // controls.pas

  PObjectInstance = ^TObjectInstance;
  TObjectInstance = packed record
    Code : Byte;
    Offset : Integer;
    case integer of
      0 : ( next   : PObjectInstance );
      1 : ( method : TAHWMethod );
  end;

  PInstanceBlock = ^TInstanceBlock;
  TInstanceBlock = packed record
    Next       : PInstanceBlock;
    Code       : array[1..2] of Byte;
    WndProcPtr : Pointer;
    Instances  : array[0..InstanceCount] of TObjectInstance;
  end;

var
  InstBlockList : PInstanceBlock;
  InstFreeList  : PObjectInstance;

{ Standard window procedure }
{ In    ECX = Address of method pointer }
{ Out   EAX = Result }

function StdWndProc( Window : HWND; Message, WParam : longint;
  LParam : longint ) : longint; stdcall; assembler;
asm
  XOR     EAX,EAX
  PUSH    EAX
  PUSH    LParam
  PUSH    WParam
  PUSH    Message
  MOV     EDX,ESP
  MOV     EAX,[ECX].longint[4]
  CALL    [ECX].Pointer
  ADD     ESP,12
  POP     EAX
end;

{ Allocate an object instance }

function CalcJmpOffset( Src, Dest: Pointer ) : longint;
begin
  result := longint( Dest ) - ( longint( Src ) + 5 );
end;

function MakeObjectInstance( Method : TAHWMethod ) : Pointer;
const
  BlockCode : array[1..2] of Byte =
    (
    $59, { POP ECX        }
    $E9  { JMP StdWndProc }
    );
  PageSize = 4096;
var
  Block    : PInstanceBlock;
  Instance : PObjectInstance;
begin
  if InstFreeList = nil then
  begin
    Block := VirtualAlloc( nil, PageSize, MEM_COMMIT, PAGE_EXECUTE_READWRITE );
    Block^.Next := InstBlockList;
    Move( BlockCode, Block^.Code, SizeOf( BlockCode ));
    Block^.WndProcPtr := Pointer( CalcJmpOffset( @Block^.Code[2], @StdWndProc ));
    Instance := @Block^.Instances;
    repeat
      Instance^.Code   := $E8;  { CALL NEAR PTR Offset }
      Instance^.Offset := CalcJmpOffset( Instance, @Block^.Code );
      Instance^.Next   := InstFreeList;

      InstFreeList := Instance;
      Inc( Longint( Instance ), SizeOf( TObjectInstance ));
    until Longint( Instance ) - Longint( Block ) >= SizeOf( TInstanceBlock );
    InstBlockList := Block;
  end;
  Result := InstFreeList;
  Instance := InstFreeList;
  InstFreeList := Instance^.Next;
  Instance^.Method := Method;
end;

{ Free an object instance }

procedure FreeObjectInstance(ObjectInstance: Pointer);
begin
  if ObjectInstance <> nil then
  begin
    PObjectInstance( ObjectInstance )^.Next := InstFreeList;
    InstFreeList := ObjectInstance;
  end;
end;

var
  UtilWindowClass : TWndClass =
    (
    style         : 0;
    lpfnWndProc   : @DefWindowProc;
    cbClsExtra    : 0;
    cbWndExtra    : 0;
    hInstance     : 0;
    hIcon         : 0;
    hCursor       : 0;
    hbrBackground : 0;
    lpszMenuName  : nil;
    lpszClassName : 'TPUtilWindow'
    );

function AllocateHWnd( Method : TAHWMethod ): HWND;
var
  TempClass       : TWndClass;
  ClassRegistered : Boolean;
begin
  UtilWindowClass.hInstance := HInstance;
  ClassRegistered := GetClassInfo
    ( HInstance, UtilWindowClass.lpszClassName, TempClass );
  if not ClassRegistered or ( TempClass.lpfnWndProc <> @DefWindowProc ) then
  begin
    if ClassRegistered then
      Windows.UnregisterClass( UtilWindowClass.lpszClassName, HInstance );
    Windows.RegisterClass( UtilWindowClass );
  end;
  Result := CreateWindowEx( WS_EX_TOOLWINDOW, UtilWindowClass.lpszClassName,
    '', WS_POPUP {!0}, 0, 0, 0, 0, 0, 0, HInstance, nil );
  if Assigned( Method ) then
    SetWindowLong( Result, GWL_WNDPROC, Longint( MakeObjectInstance( Method )));
end;

procedure DeallocateHWnd( Wnd : HWND );
var
  Instance : Pointer;
begin
  Instance := Pointer( GetWindowLong( Wnd, GWL_WNDPROC ));
  DestroyWindow( Wnd );
  if Instance <> @DefWindowProc then FreeObjectInstance( Instance );
end;


// -----------------------------------------------------------------------------
// -----------------------------------------------------------------------------
// -----------------------------------------------------------------------------

constructor TmePegasusControl.create( aOwner: TComponent );
begin
  inherited;
  fWindowHandle := AllocateHWnd( WndProc );
  LoadDefaults;
  fPegasusProtocol := TPegasusProtocol.create( fWindowHandle );

  fHeartbeat := TTimer.create( nil );
  with fHeartbeat do
  begin
    Interval := HB_INTERVAL;
    OnTimer  := DoHeartbeat;
    Enabled  := false;
  end;
end;

destructor TmePegasusControl.destroy;
begin
  if assigned( fPegasusProtocol ) and fControlRec.bMuteOnExit then
  with fPegasusProtocol do
  begin
    SpeakerOutput( 0 );
    LineLevelAudioOutput( 63 );
  end;
  fPegasusProtocol.free;
  fHeartBeat.free;
  DeallocateHWnd( fWindowHandle );
  inherited;
end;

procedure TmePegasusControl.WndProc( var aMsg : TMessage );
begin
  with aMsg do
  case Msg of
    PM_DSPSTART   : DoDSPStart;
    PM_RADIOSTART : DoRadioStart;
    PM_ENCODER    : DoEncoder( WParam, LParam );
    PM_FORWARD    : DoForward( WParam );
    PM_REFLECTED  : DoReflected( WParam );
    PM_RXMETER    : DoRxMeter( WParam );
    PM_TXMETER    : DoTxMeter( WParam, LParam );
    PM_PODKEY     : DoPodKey( WParam );
    PM_VERSION    : DoVersion( WParam );
    PM_DIGLEVEL   : DoDigLevel( WParam );
    PM_ANALEVEL   : DoAnaLevel( WParam );
    PM_BADSYNTAX  : DoBadSyntax;
  end;
end;

procedure TmePegasusControl.DoDSPStart;
begin
  fPegasusProtocol.DSPProgramExecute;
  fNeedRestart := false;
end;

procedure TmePegasusControl.DoRadioStart;
begin
  fPegasusProtocol.QueryVersion;
  fNeedRestart := false;
  SetupRadio;
end;

procedure TmePegasusControl.DoEncoder( const aMovement, aKeyPressed : integer );
begin
  fLastEncoder := aMovement;
  fLastEncoderKey := chr( aKeyPressed );
  DoPropertyChange( wpEncoder );
end;

procedure TmePegasusControl.DoForward( const aForward : integer );
begin
  fLastForward := aForward;
  DoPropertyChange( wpForwardpower );
end;

procedure TmePegasusControl.DoReflected( const aReflected : integer );
begin
  fLastReflected := aReflected;
  DoPropertyChange( wpReflectedPower );
end;

procedure TmePegasusControl.DoRxMeter( const aSMeter : integer );
begin
  fSMeterFrequency := fVFORec.bRxFrequency;
  fLastSMeter := aSMeter;


  if fLastStateTx then
  begin
    fLastStateTx := false;
    CheckMonitorBoost;
    DoPropertyChange( wpStartRx );
  end;

  DoPropertyChange( wpRxMeter );
end;

procedure TmePegasusControl.DoTxMeter( const aForward, aReflected : integer );
begin
  fLastForward := aForward;
  fLastReflected := aReflected;
  if not fLastStateTx then
  begin
    fLastStateTx := true;
    CheckMonitorBoost;
    DoPropertyChange( wpStartTx );
  end;

  DoPropertyChange( wpTxMeter );
end;

procedure TmePegasusControl.DoPodKey( const aKeyPressed : integer );
begin
  fLastPodKeypress := chr( aKeyPressed );
  DoPropertyChange( wpPodKeypress );
end;

procedure TmePegasusControl.DoVersion( const aVersion : integer );
begin
  fDSPVersion := aVersion;
end;

procedure TmePegasusControl.DoDigLevel( const aLevel : integer );
begin
  fLastDSPSignalLevel := aLevel;
  DoPropertyChange( wpDSPSignalLevel );
end;

procedure TmePegasusControl.DoAnaLevel( const aLevel : integer );
begin
  fLastAnalogAGCLevel := aLevel;
  DoPropertyChange( wpAnalogAGCLevel );
end;

procedure TmePegasusControl.DoBadSyntax;
begin
  DoPropertyChange( wpBadSyntax );
end;

// -------------------------------------------------------------------

procedure TmePegasusControl.SetComPort( const value : byte );
begin
  // if fComPort <> value then
  begin
    if fComPort > 0 then
    begin
      if fControlRec.bMuteOnExit then
        fPegasusProtocol.SpeakerOutput( 0 );
      fPegasusProtocol.CloseSerialPort;
      fHeartbeat.Enabled := false;
    end;
    fComPort := value;
    if fComPort > 0 then
    begin
      fPegasusProtocol.OpenSerialPort( fComPort );
      fHeartbeat.Enabled := true;
      fNeedRestart := true;
    end;
    DoPropertyChange( wpComPort );
  end;
end;

procedure TmePegasusControl.DoHeartbeat( sender : TObject );
begin
  if fNeedRestart
    then fPegasusProtocol.RestartAndNotify
  else
  begin
    fPegasusProtocol.QuerySignalLevel;

    if fTunerActive then DoTunerHeartbeat;
  end;

  DoPropertyChange( wpHeartbeat );
end;

procedure TmePegasusControl.DoTunerHeartbeat;

  procedure EndTune;
  begin
    with fPegasusProtocol do
    begin
      KeyDown( false );
      BeginUpdateVFO;
      fVFORec.bRxMode := fTunerHoldRxMode;
      fVFORec.bTxMode := fTunerHoldTxMode;
      EndUpdateVFO;
      OutputPower( fControlRec.bRFOut );
    end;
    fTunerStep := tunerNone;
    fTunerActive := false;

    DoPropertyChange( wpTunerDone );
  end;

begin { procedure DoTunerHeartbeat }
  case fTunerStep of
    tunerRFOn :
      with fPegasusProtocol do
      begin
        fTunerHoldRxMode := fVFORec.bRxMode;
        fTunerHoldTxMode := fVFORec.bTxMode;
        BeginUpdateVFO;
        fVFORec.bRxMode := MODE_CW;
        fVFORec.bTxMode := MODE_CW;
        EndUpdateVFO;
        if fTunerCommand = MEASURE_SWR then
        begin
          OutputPower( fControlRec.bSWRRFOut );
          fTunerStep := tunerSWR;
          fTunerCtr := TUNER_SWR_PAUSE;
        end
        else
        begin
          OutputPower( fControlRec.bTunerRFOut );
          fTunerStep := tunerCmd;
        end;
        KeyDown( true );
        if fTunerCommand = AT11_TUNE
          then SetDTR( true );
      end;
    tunerCmd :
      with fPegasusProtocol do
      begin
        if fTunerCommand = AT11_TUNE
          then SetDTR( false )
          else fPegasusProtocol.PT11Tuner( fTunerCommand );
        fTunerCtr := TUNER_TIMEOUT;
        fTunerDupCtr := TUNER_DUPCTR;
        fTunerStep := tunerWait
      end;
    tunerWait :
      begin
        dec( fTunerCtr );
        if fTunerCtr <= 0 then
        begin
          fTunerOK := false;
          EndTune
        end
        else
        begin
          if ( abs( fLastForward - fTunerFwd ) <= TUNER_MAX_FWD_DELTA )
            and ( abs( fLastReflected - fTunerRef ) <= TUNER_MAX_REF_DELTA ) then
          begin
            dec( fTunerDupCtr );
            if fTunerDupCtr <= 0 then
            begin
              fTunerOK := true;
              EndTune;
            end;
          end
          else
          begin
            fTunerDupCtr := TUNER_DUPCTR;
            fTunerFwd := fLastForward;
            fTunerRef := fLastReflected;
          end;
        end;
      end;
    tunerSWR :
      begin
        dec( fTunerCtr );
        if fTunerCtr <= 0 then
        begin
          fTunerOK := true;
          EndTune;
        end;
      end;
  else
    EndTune;
  end;
end;

procedure TmePegasusControl.CheckMonitorBoost;
begin
  if ( fControlRec.bMonitorBoost <> MONITOR_BOOST_NONE ) and ( fVFORec.bTxMode <> MODE_CW ) then
  begin
    if fLastStateTx then
    begin
      fPegasusProtocol.SpeakerOutput( 255 );
      if fControlRec.bMonitorBoost = MONITOR_BOOST_BOTH then
        fPegasusProtocol.LineLevelAudioOutput( 0 );
    end
    else
    begin
      fPegasusProtocol.SpeakerOutput( fControlRec.bAFGain );
      fPegasusProtocol.LineLevelAudioOutput( fControlRec.bLineLevel );
    end;
  end;
end;

{ --------------------------------------------------------------------------- }
{ ----- group command methods ----------------------------------------------- }
{ --------------------------------------------------------------------------- }

{
  This is needed as some of the commands to the Pegasus depend on multiple
  parameters or are affected by other parameters. Without grouping, each time
  a constituent parameter was changed, commands would have to be sent to
  the Pegasus instead of when all the related parameters had been changed.

  To use call the appropriate BeginUpdate... method before changing the
  related parameters and call EndUpdate... afterwards. If only one parameter
  in a group is being changed, there is no need to call BeginUpdate ...
  EndUpdate...
}


{ --------------------------------------------------------------------------- }
{ ---- InputSourceGain ------------------------------------------------------ }
{ --------------------------------------------------------------------------- }

// parameters involved: MicGain, AuxGain, AuxInput
// required because all three are involved in a single command to the Pegasus

procedure TmePegasusControl.BeginUpdateInputSourceGain;
begin
  if not fInUpdateInputSourceGain then
  begin
    fInUpdateInputSourceGain := true;
    fHoldAuxGain := fControlRec.bAuxGain;
    fHoldMicGain := fControlRec.bMicGain;
    fHoldAuxInput := fControlRec.bAuxInput;
  end;
end;

procedure TmePegasusControl.EndUpdateInputSourceGain;
var
  gain, holdGain : byte;
begin
  if fInUpdateInputSourceGain then
  begin
    fInUpdateInputSourceGain := false;
    if fControlRec.bAuxInput then
    begin
      gain := fControlRec.bAuxGain;
      holdGain := fHoldAuxGain;
    end
    else
    begin
      gain := fControlRec.bMicGain;
      holdGain := fHoldMicGain;
    end;

    if (( fControlRec.bAuxInput <> fHoldAuxInput ) or ( gain <> holdGain )) then
      fPegasusProtocol.AudioSourceAndGain( fControlRec.bAuxInput, gain );
  end;
end;


{ --------------------------------------------------------------------------- }
{ ----- NRAutoNotch --------------------------------------------------------- }
{ --------------------------------------------------------------------------- }

// parameters involved: NR, AutoNotch
// required because both are involved in a single command to the Pegasus

procedure TmePegasusControl.BeginUpdateNRAutoNotch;
begin
  if not fInUpdateNRAutoNotch then
  begin
    fInUpdateNRAutoNotch := true;
    fHoldNR := fControlRec.bNR;
    fHoldAutoNotch := fControlRec.bAutoNotch;
  end;
end;

procedure TmePegasusControl.EndUpdateNRAutoNotch;
begin
  if fInUpdateNRAutoNotch then
  begin
    if (( fHoldNR <> fControlRec.bNR ) or ( fHoldAutoNotch <> fControlRec.bAutoNotch )) then
      fPegasusProtocol.NoiseReductionAutoNotch( fControlRec.bNR, fControlRec.bAutoNotch );
  end;
end;


{ --------------------------------------------------------------------------- }
{ ----- Keyer --------------------------------------------------------------- }
{ --------------------------------------------------------------------------- }

// parameters involved keyer speed, keyer weight, paddle reverse
// required because all three are involved in a single command to the Pegasus

procedure TmePegasusControl.BeginUpdateKeyer;
begin
  if not fInUpdateKeyer then
  begin
    fInUpdateKeyer := true;
    fHoldKeyerSpeed := fControlRec.bKeyerSpeed;
    fHoldKeyerWeight := fControlRec.bKeyerWeight;
    fHoldPaddleReverse := fControlRec.bPaddleReverse;
  end;
end;

procedure TmePegasusControl.EndUpdateKeyer;
begin
  if fInUpdateKeyer then
  begin
    fInUpdateKeyer := false;
    if ( fControlRec.bKeyerSpeed <> fHoldKeyerSpeed )
      or ( fControlRec.bKeyerWeight <> fHoldKeyerWeight )
      or ( fControlRec.bPaddleReverse <> fHoldPaddleReverse ) then
        KeyerParams;
  end;
end;


{ --------------------------------------------------------------------------- }
{ ------ VFO ---------------------------------------------------------------- }
{ --------------------------------------------------------------------------- }

// parameters involved: CWPitch, RxMode, RxFrequency, RxFilter, RxPBTLevel,
//   TxMode, TxFrequency, TxFilter, TxPBTLevel

// required as the CWPitch and Rx factors interract in the Rx tuning parameters,
//   CWPitch and the Tx factors interract in the Rx tuning parameters,
//   and RxMode and TxMode are combined into a single command to the Pegasus

procedure TmePegasusControl.BeginUpdateVFO;
begin
  if not fInUpdateVFO then
  begin
    fInUpdateVFO := true;
    fHoldVFO := fVFORec;
  end;
end;

procedure TmePegasusControl.EndUpdateVFO;
var
  newFilter : Boolean;
begin
  if fInUpdateVFO then
    with fPegasusProtocol do
    begin
      fInUpdateVFO := false;
      if ( fHoldVFO.bRxMode <> fVFORec.bRxMode ) or ( fHoldVFO.bTxMode <> fVFORec.bTxMode ) then
        RxTxModesInt( fVFORec.bRxMode, fVFORec.bTxMode );
      newFilter := fHoldVFO.bRxFilter <> fVFORec.bRxFilter;
      if newFilter then
        RxFilter( AdjustFilter( fVFORec.bRxFilter ));
      if newFilter
        or ( fHoldVFO.bRxMode <> fVFORec.bRxMode )
        or ( CalcAdjFrequency( fHoldVFO.bRIT, fHoldVFO.bRxFrequency, fHoldVFO.bRITLevel ) <>
             CalcAdjFrequency( fVFORec.bRIT, fVFORec.bRxFrequency, fVFORec.bRITLevel ))
        or ( CalcOnOffInteger( fHoldVFO.bRxPBT, fHoldVFO.bRxPBTLevel ) <>
             CalcOnOffInteger( fVFORec.bRxPBT, fVFORec.bRxPBTLevel ))
        or (( fVFORec.bRxMode = MODE_CW ) and ( fHoldVFO.bCWPitch <> fVFORec.bCWPitch )) then
      UpdateRxParams;
      newFilter := fHoldVFO.bTxFilter <> fVFORec.bTxFilter;
      if newFilter then
        TxFilter( AdjustFilter( fVFORec.bTxFilter ));
      if newFilter
        or ( fHoldVFO.bTxMode <> fVFORec.bTxMode )
        or ( CalcAdjFrequency( fHoldVFO.bXIT, fHoldVFO.bTxFrequency, fHoldVFO.bXITLevel ) <>
             CalcAdjFrequency( fVFORec.bXIT, fVFORec.bTxFrequency, fVFORec.bXITLevel ))
        or ( CalcOnOffInteger( fHoldVFO.bTxPBT, fHoldVFO.bTxPBTLevel ) <>
             CalcOnOffInteger( fVFORec.bTxPBT, fVFORec.bTxPBTLevel ))
        or (( fVFORec.bTxMode = MODE_CW ) and ( fHoldVFO.bCWPitch <> fVFORec.bCWPitch )) then
      UpdateTxParams;
    end;
end;


{ --------------------------------------------------------------------------- }
{ ----- external support methods -------------------------------------------- }
{ --------------------------------------------------------------------------- }

procedure TmePegasusControl.SetupKeyer(const aSpeed, aWeight: byte; const aReverse: Boolean);
begin
  if ( aSpeed <> fControlRec.bKeyerSpeed ) or ( aWeight <> fControlRec.bKeyerWeight )
    or ( aReverse <> fControlRec.bPaddleReverse ) then
  begin
    BeginUpdateKeyer;
    KeyerSpeed := aSpeed;
    KeyerWeight := aWeight;
    PaddleReverse := aReverse;
    EndUpdateKeyer;
  end;
end;

function TmePegasusControl.GetLastSWR : double;
var
  rho : double;
begin
  if ( fLastForward = 0 ) or ( fLastForward = fLastReflected ) then
    result := 999.9
  else
  begin
    rho := sqrt( fLastReflected / fLastForward );
    result := ( 1.0 + rho ) / ( 1.0 - rho )
  end;
end;


{ --------------------------------------------------------------------------- }
{ ----- internal support methods -------------------------------------------- }
{ --------------------------------------------------------------------------- }

procedure TmePegasusControl.KeyerParams;
var
  dit, dah, space : word;
begin
  dit := 7200 div fControlRec.bKeyerSpeed;
  space := MulDiv( dit, fControlRec.bKeyerWeight, 50 );
  dah := 2 * dit + space;
  with fPegasusProtocol do
    if fControlRec.bPaddleReverse
      then KeyerTiming( dah, dit, space )
      else KeyerTiming( dit, dah, space );
end;

procedure TmePegasusControl.UpdateRxParams;
const
  MIN_SSB = 200;
  MIN_CW  = 300;
var
  coarse, fine, bfo : word;
  tFreq, iBFO, FilterBW2 : integer;
  adjPBTLevel : integer;
  adjFrequency : integer;
begin
  if fVFORec.bRxPBT
    then adjPBTLevel := fVFORec.bRxPBTLevel
    else adjPBTLevel := 0;
  adjFrequency := CalcAdjFrequency( fVFORec.bRIT, fVFORec.bRxFrequency, fVFORec.bRITLevel );
  FilterBW2 := FILTER_WIDTH[fVFORec.bRxFilter] shr 1;
  case fVFORec.bRxMode of
    MODE_USB, MODE_LSB :
      begin
        iBFO  := FilterBW2 + MIN_SSB + adjPBTLevel;
        if fVFORec.bRxMode = MODE_USB
          then tFreq := adjFrequency + iBFO
          else tFreq := adjFrequency - iBFO;
      end;
    MODE_CW :
      begin
        if FilterBW2 + MIN_CW <= fVFORec.bCWPitch then
        begin
          iBFO  := fVFORec.bCWPitch + adjPBTLevel;
          tFreq := adjFrequency - adjPBTLevel;
        end
        else
        begin
          iBFO  := FilterBW2 + MIN_CW + adjPBTLevel;
          tFreq := adjFrequency - iBFO + fVFORec.bCWPitch;
        end;
      end;
    else
      begin
        iBFO  := 0;
        tFreq := adjFrequency;
      end
  end;

  dec( tFreq, 1250 );
  coarse := tFreq div 2500 + 18000;
  fine   := (( tFreq mod 2500 ) shl 16 ) div 12000;
  BFO    := (( iBFO + 8000 ) shl 16 ) div 24000;

  fPegasusProtocol.RxTuningFactors( coarse, fine, BFO );
end;

procedure TmePegasusControl.UpdateTxParams;
const
  FIXED_BFO = 1500;
  MIN_SSB   = 200;
var
  tFreq : longint;
  filterBW2, iBFO, bwBFO : smallint;
  coarse, fine, BFO : word;
  // adjPBT : integer;
  adjFrequency : integer;
begin
  {
  if fTxPBT
    then adjPBT := fTxPBTLevel
    else adjPBT := 0;
  }
  adjFrequency := CalcAdjFrequency( fVFORec.bXIT, fVFORec.bTxFrequency, fVFORec.bXITLevel );
  case fVFORec.bTxMode of
    MODE_USB, MODE_LSB :
      begin
        filterBW2 := FILTER_WIDTH[fVFORec.bTxFilter] shr 1;
        iBFO  := FIXED_BFO;   // SSB fixed Tx for now (comment in TT source)
        bwBFO := filterBW2 + MIN_SSB;
        if bwBFO > iBFO
          then iBFO := bwBFO;
        if fVFORec.bTxMode = MODE_USB
          then tFreq := adjFrequency + iBFO
          else tFreq := adjFrequency - iBFO;
      end;
    MODE_CW :
      begin
        // CW mode uses LSB mode
        iBFO  := FIXED_BFO;
        tFreq := adjFrequency - iBFO + fVFORec.bCWPitch;
        iBFO  := fVFORec.bCWPitch;
      end;
    else
      begin
        iBFO  := 0;
        tFreq := adjFrequency;
      end;
  end;

  dec( tFreq, 1250 );
  coarse := tFreq div 2500 + 18000;
  fine   := (( tFreq mod 2500 ) shl 16 ) div 12000;
  BFO    := ( iBFO shl 16 ) div 24000;

  fPegasusProtocol.TxTuningFactors( coarse, fine, BFO );
end;


procedure TmePegasusControl.SetupRadio;
var
  wp : TWhichProperty;
begin
  with fPegasusProtocol do
  begin
    fDoNotify := false;
    fForceUpdate := true;
    SpeakerOutput( 0 );
    RxTxModesInt( fVFORec.bRxMode, fVFORec.bTxMode );
    RxFilter( AdjustFilter( fVFORec.bRxFilter ));
    TxFilter( AdjustFilter( fVFORec.bTxFilter ));
    UpdateRxParams;
    UpdateTxParams;
    LineLevelAudioOutput( fControlRec.bLineLevel );
    AGCMode( chr( fControlRec.bAGC + ord( '1' )));
    RFGain( fControlRec.bRFGain );
    RFAttenuator( fControlRec.bRFAtten );
    KeyerParams;

    SetOnOffByte( fControlRec.bMonitor, fControlRec.bMonitorLevel, TxAudioMonitorVolume );
    SetOnOffByte( fControlRec.bSidetone, fControlRec.bSidetoneLevel, CWSidetoneVolume );

    NoiseReductionAutoNotch( fControlRec.bNR, fControlRec.bAutoNotch );

    if fControlRec.bAuxInput
      then AudioSourceAndGain( true, fControlRec.bAuxGain )
      else AudioSourceAndGain( false, fControlRec.bMicGain );
    OutputPower( fControlRec.bRFOut );
    KeyDown( false );

    SetOnOffByte( fControlRec.bSquelch, fControlRec.bSquelchLevel, Squelch );

    VOX( fControlRec.bVOX );
    VOXGain( fControlRec.bVOXGain );
    VOXAnti( fControlRec.bVOXAnti );
    VOXHang( fControlRec.bVOXHang );

    SetOnOffByte( fControlRec.bSpot, fControlRec.bSpotLevel, CWSpotLevel );

    Transmitter( fControlRec.bTxEnable );
    TLoop( fControlRec.bTLoop );
    WideBandTx( fControlRec.bWidebandTx );
    Keyer( fControlRec.bKeyerEnable );
    KeepAlive( fControlRec.bTimeout );

    SetOnOffByte( fControlRec.bSpeechProcessor, fControlRec.bSpeechProcessorLevel, SpeechProcessor );
    SetOnOffByte( fControlRec.bNB, fControlRec.bNBLevel, NoiseBlankerLevel );

    SetOnOffByte( not fControlRec.bMute, fControlRec.bAFGain, SpeakerOutput );
    fDoNotify := true;
    fForceUpdate := false;
  end;

  for wp := wpCOMPort to high( TWhichProperty ) do
    DoPropertyChange( wp );
end;

procedure TmePegasusControl.Tuner( const aWhich : char );
begin
  fTunerStep := tunerRFOn;
  fTunerCommand := aWhich;
  fTunerActive := true;
end;

procedure TmePegasusControl.DoPropertyChange( const aWhichProperty : TWhichProperty );
begin
  try
    fInUpdateNotify := true;
    if assigned( fOnChangeArray[aWhichProperty] ) then
      fOnChangeArray[aWhichProperty]( self );
  finally
    fInUpdateNotify := false;
  end;
end;

class function TmePegasusControl.AdjustFilter( const aValue : byte ) : byte;
begin
  // handle the fact that the Pegasus filters are ordered 6000 .. 300, 8000
  if aValue = 0
    then result := 33
    else result := aValue - 1;
end;


{ --------------------------------------------------------------------------- }
{ ----- property get / set methods ------------------------------------------ }
{ --------------------------------------------------------------------------- }


procedure TmePegasusControl.SetCurrentVFO( const value : TVFORec );
begin
  if not compareMem( @fVFORec, @value, sizeof( TVFORec )) then
  begin
    fVFORec := value;
    BeginUpdateVFO;
    with value do
    begin
      CWPitch     := bCWPitch;
      RxMode      := bRxMode;
      RxFrequency := bRxFrequency;
      RxFilter    := bRxFilter;
      RxPBT       := bRxPBT;
      RxPBTLevel  := bRxPBTLevel;
      RIT         := bRIT;
      RITLevel    := bRITLevel;
      TxMode      := bTxMode;
      TxFrequency := bTxFrequency;
      TxFilter    := bTxFilter;
      TxPBT       := bTxPBT;
      TxPBTLevel  := bTxPBTLevel;
      XIT         := bXIT;
      XITLevel    := bXITLevel;
    end;
    EndUpdateVFO;
  end;
end;

procedure TmePegasusControl.SetCurrentParams( const value : TPegasusControlRec );
begin
  if not compareMem( @fControlRec, @value, sizeof( TPegasusControlRec )) then
  begin
    with value do
    begin
      BeginUpdateNRAutoNotch;
        NR := bNR;
        AutoNotch := bAutoNotch;
      EndUpdateNRAutoNotch;

      BeginUpdateInputSourceGain;
        AuxGain  := bAuxGain;
        MicGain  := bMicGain;
        AuxInput := bAuxInput;
      EndUpdateInputSourceGain;

      SetupKeyer( bKeyerSpeed, bKeyerWeight, bPaddleReverse );

      KeyerEnable          := bKeyerEnable;

      LineLevel            := bLineLevel;
      Mute                 := bMute;
      MuteOnExit           := bMuteOnExit;
      AFGain               := bAFGain;
      Sidetone             := bSidetone;
      SidetoneLevel        := bSidetoneLevel;
      Spot                 := bSpot;
      SpotLevel            := bSpotLevel;
      Monitor              := bMonitor;
      MonitorLevel         := bMonitorLevel;
      MonitorBoost         := bMonitorBoost;

      RFGain               := bRFGain;
      RFAtten              := bRFAtten;
      AGC                  := bAGC;
      NB                   := bNB;
      NBLevel              := bNBLevel;
      Squelch              := bSquelch;
      SquelchLevel         := bSquelchLevel;

      RFOut                := bRFOut;
      QSK                  := bQSK;
      TxHang               := bTxHang;
      TLoop                := bTLoop;
      TxEnable             := bTxEnable;
      KeyDown              := bKeyDown;
      Timeout              := bTimeout;
      WidebandTx           := bWidebandTx;
      SpeechProcessor      := bSpeechProcessor;
      SpeechProcessorLevel := bSpeechProcessorLevel;
      AMCarrier            := bAMCarrier;

      VOX                  := bVOX;
      VOXGain              := bVOXGain;
      VOXAnti              := bVOXAnti;
      VOXHang              := bVOXHang;

      TunerRFOut           := bTunerRFOut;

      SWRRFOut             := bSWRRFOut;
    end;
  end;
end;


procedure TmePegasusControl.SetAFGain(const Value: byte);
begin
  if fControlRec.bAFGain <> Value then
  begin
    fControlRec.bAFGain := Value;
    SetOnOffByte( not fControlRec.bMute, fControlRec.bAFGain, fPegasusProtocol.SpeakerOutput );
    DoPropertyChange( wpAFGain );
  end;
end;

procedure TmePegasusControl.SetAGC(const Value: byte);
begin
  if fControlRec.bAGC <> Value then
  begin
    fControlRec.bAGC := Value;
    fPegasusProtocol.AGCMode( chr( fControlRec.bAGC + ord( '1' )));
    DoPropertyChange( wpAGC );
  end;
end;

procedure TmePegasusControl.SetAMCarrier(const Value: byte);
begin
  if fControlRec.bAMCarrier <> Value then
  begin
    fControlRec.bAMCarrier := Value;
    fPegasusProtocol.AMCarrier( fControlRec.bAMCarrier );
    DoPropertyChange( wpAMCarrier );
  end;
end;

procedure TmePegasusControl.SetAutoNotch(const Value: Boolean);
begin
  if fControlRec.bAutoNotch <> Value then
  begin
    fControlRec.bAutoNotch := Value;
    if not fInUpdateNRAutoNotch then
      fPegasusProtocol.NoiseReductionAutoNotch( fControlRec.bNR, fControlRec.bAutoNotch );
    DoPropertyChange( wpAutoNotch );
  end;
end;

procedure TmePegasusControl.SetAuxGain(const Value: byte);
begin
  if fControlRec.bAuxGain <> Value then
  begin
    fControlRec.bAuxGain := Value;
    if fControlRec.bAuxInput and not fInUpdateInputSourceGain then
      fPegasusProtocol.AudioSourceAndGain( true, fControlRec.bAuxGain );
    DoPropertyChange( wpAuxGain );
  end;
end;

procedure TmePegasusControl.SetAuxInput(const Value: Boolean);
var
  gain : byte;
begin
  if fControlRec.bAuxInput <> Value then
  begin
    fControlRec.bAuxInput := Value;
    if not fInUpdateInputSourceGain then
    begin
      if fControlRec.bAuxInput
        then gain := fControlRec.bAuxGain
        else gain := fControlRec.bMicGain;
      fPegasusProtocol.AudioSourceAndGain( fControlRec.bAuxInput, gain );
    end;
    DoPropertyChange( wpAuxInput );
  end;
end;

procedure TmePegasusControl.SetCWPitch(const Value: word);
begin
  if fVFORec.bCWPitch <> Value then
  begin
    fVFORec.bCWPitch := Value;
    if not fInUpdateVFO then
    with fPegasusProtocol do
    begin
      if fVFORec.bRxMode = MODE_CW then UpdateRxParams;
      if fVFORec.bTxMode = MODE_CW then UpdateTxParams;
    end;
    DoPropertyChange( wpCWPitch );
  end;
end;

procedure TmePegasusControl.SetKeyDown(const Value: Boolean);
begin
  if fControlRec.bKeyDown <> Value then
  begin
    fControlRec.bKeyDown := Value;
    fPegasusProtocol.KeyDown( fControlRec.bKeyDown );
    DoPropertyChange( wpKeyDown );
  end;
end;

procedure TmePegasusControl.SetKeyerEnable(const Value: Boolean);
begin
  if fControlRec.bKeyerEnable <> value then
  begin
    fControlRec.bKeyerEnable := Value;
    fPegasusProtocol.Keyer( fControlRec.bKeyerEnable );
    DoPropertyChange( wpKeyerEnable );
  end;
end;

procedure TmePegasusControl.SetLineLevel(const Value: byte);
begin
  if fControlRec.bLineLevel <> value then
  begin
    fControlRec.bLineLevel := Value;
    fPegasusProtocol.LineLevelAudioOutput( fControlRec.bLineLevel );
    DoPropertyChange( wpLineLevel );
  end;
end;

procedure TmePegasusControl.SetMicGain(const Value: byte);
begin
  if fControlRec.bMicGain <> Value then
  begin
    fControlRec.bMicGain := Value;
    if not ( fControlRec.bAuxInput or fInUpdateInputSourceGain ) then
      fPegasusProtocol.AudioSourceAndGain( false, fControlRec.bMicGain );
    DoPropertyChange( wpMicGain );
  end;
end;

class function TmePegasusControl.CalcOnOffInteger( const aOn : Boolean; const aValue : integer ) : integer;
begin
  if aOn
    then result := aValue
    else result := 0;
end;

class procedure TmePegasusControl.SetOnOffByte( const aOn : Boolean; const aLevel : byte;
  const aWhich : TPegPSetByte );
begin
  aWhich( calcOnOffInteger( aOn, aLevel ) );
end;

class function TmePegasusControl.CalcAdjFrequency( const aOn : Boolean; const aFrequency, aAdjust : integer ) : integer;
begin
  result := aFrequency + CalcOnOffInteger( aOn, aAdjust );
end;

procedure TmePegasusControl.SetMonitor(const Value: Boolean);
begin
  if fControlRec.bMonitor <> value then
  begin
    fControlRec.bMonitor := Value;
    SetOnOffByte( fControlRec.bMonitor, fControlRec.bMonitorLevel, fPegasusProtocol.TxAudioMonitorVolume );
    DoPropertyChange( wpMonitor );
  end;
end;

// -----------------------------------------------------------------------------

procedure TmePegasusControl.RxTxModesInt( const aRxMode, aTxMode : byte );
begin
  fPegasusProtocol.RxTxModes( chr( aRxMode + ord( '0' )), chr( aTxMode + ord( '0' )));
end;


procedure TmePegasusControl.SetMonitorLevel(const Value: byte);
begin
  if fControlRec.bMonitorLevel <> value then
  begin
    fControlRec.bMonitorLevel := Value;
    SetOnOffByte( fControlRec.bMonitor, fControlRec.bMonitorLevel, fPegasusProtocol.TxAudioMonitorVolume );
    DoPropertyChange( wpMonitorLevel );
  end;
end;

// -----------------------------------------------------------------------------

procedure TmePegasusControl.SetMonitorBoost( const value : byte );
begin
  if fControlRec.bMonitorBoost <> value then
  begin
    fControlRec.bMonitorBoost := value;
    DoPropertyChange( wpMonitorBoost );
  end;
end;

procedure TmePegasusControl.SetMute(const Value: Boolean);
begin
  if fControlRec.bMute <> Value then
  begin
    fControlRec.bMute := Value;
    SetOnOffByte( not fControlRec.bMute, fControlRec.bAFGain, fPegasusProtocol.SpeakerOutput );
    DoPropertyChange( wpMute );
  end;
end;

procedure TmePegasusControl.SetNBLevel(const Value: byte);
begin
  if fControlRec.bNBLevel <> Value then
  begin
    fControlRec.bNBLevel := Value;
    SetOnOffByte( fControlRec.bNB, fControlRec.bNBLevel, fPegasusProtocol.NoiseBlankerLevel );
    DoPropertyChange( wpNBLevel );
  end;
end;

procedure TmePegasusControl.SetNB( const Value : Boolean );
begin
  if fControlRec.bNB <> Value then
  begin
    fControlRec.bNB := Value;
    SetOnOffByte( fControlRec.bNB, fControlRec.bNBLevel, fPegasusProtocol.NoiseBlankerLevel );
    DoPropertyChange( wpNB );
  end;
end;

procedure TmePegasusControl.SetNR(const Value: Boolean);
begin
  if fControlRec.bNR <> Value then
  begin
    fControlRec.bNR := Value;
    if not fInUpdateNRAutoNotch then
      fPegasusProtocol.NoiseReductionAutoNotch( fControlRec.bNR, fControlRec.bAutoNotch );
    DoPropertyChange( wpNR );
  end;
end;

procedure TmePegasusControl.SetQSK(const Value: byte);
begin
  if fControlRec.bQSK <> Value then
  begin
    fControlRec.bQSK := Value;
    fPegasusProtocol.CWQSK( fControlRec.bQSK );
    DoPropertyChange( wpQSK );
  end;
end;

procedure TmePegasusControl.SetRFAtten(const Value: Boolean);
begin
  if fControlRec.bRFAtten <> Value then
  begin
    with fPegasusProtocol do
    begin
      fControlRec.bRFAtten := Value;
      fPegasusProtocol.RFAttenuator( fControlRec.bRFAtten );
    end;
    DoPropertyChange( wpRFAtten );
  end;
end;

procedure TmePegasusControl.SetRFGain(const Value: byte);
begin
  if fControlRec.bRFGain <> Value then
  begin
    fControlRec.bRFGain := Value;
    fPegasusProtocol.RFGain( fControlRec.bRFGain );
    DoPropertyChange( wpRFGain );
  end;
end;

procedure TmePegasusControl.SetRFOut(const Value: byte);
begin
  if fControlRec.bRFOut <> Value then
  begin
    fControlRec.bRFOut := Value;
    fPegasusProtocol.OutputPower( fControlRec.bRFOut );
    DoPropertyChange( wpRFOut );
  end;
end;

procedure TmePegasusControl.SetRxFilter(const Value: byte);
begin
  if fVFORec.bRxFilter <> Value then
  begin
    fVFORec.bRxFilter := Value;
    if not fInUpdateVFO then
    begin
      fPegasusProtocol.RxFilter( AdjustFilter( fVFORec.bRxFilter ));
      UpdateRxParams
    end;
    DoPropertyChange( wpRXFilter );
  end;
end;

procedure TmePegasusControl.SetRxFrequency(const Value: integer);
begin
  if fVFORec.bRxFrequency <> Value then
  begin
    fVFORec.bRxFrequency := Value;
    if not fInUpdateVFO then
      UpdateRxParams;
    DoPropertyChange( wpRxFrequency );
  end;
end;

procedure TmePegasusControl.SetRxMode(const Value: byte);
begin
  if fVFORec.bRxMode <> Value then
  begin
    fVFORec.bRxMode := Value;
    if not fInUpdateVFO then
    begin
      RxTxModesInt( fVFORec.bRxMode, fVFORec.bTxMode );
      UpdateRxParams;
    end;
    DoPropertyChange( wpRxMode );
  end;
end;

procedure TmePegasusControl.SetRxPBT( const value : Boolean );
begin
  if fVFORec.bRxPBT <> value then
  begin
    fVFORec.bRxPBT := value;
    if not fInUpdateVFO then
      UpdateRxParams;
    DoPropertyChange( wpRxPBT );
  end;
end;

procedure TmePegasusControl.SetRxPBTLevel(const Value: smallint);
begin
  if fVFORec.bRxPBTLevel <> Value then
  begin
    fVFORec.bRxPBTLevel := Value;
    if not fInUpdateVFO then
      UpdateRxParams;
    DoPropertyChange( wpRxPBTLevel );
  end;
end;

procedure TmePegasusControl.SetRIT( const value : Boolean );
begin
  if fVFORec.bRIT <> value then
  begin
    fVFORec.bRIT := value;
    if not fInUpdateVFO then
      UpdateRxParams;
    DoPropertyChange( wpRIT );
  end;
end;

procedure TmePegasusControl.SetRITLevel( const value : smallint );
begin
  if fVFORec.bRITLevel <> value  then
  begin
    fVFORec.bRITLevel := value;
    if not fInUpdateVFO then
      UpdateRxParams;
    DoPropertyChange( wpRITLevel );
  end;
end;

procedure TmePegasusControl.SetSidetone(const Value: Boolean);
begin
  if fControlRec.bSidetone <> value then
  begin
    fControlRec.bSidetone := value;
    SetOnOffByte( fControlRec.bSidetone, fControlRec.bSidetoneLevel, fPegasusProtocol.CWSidetoneVolume );
    DoPropertyChange( wpSidetone );
  end;
end;

procedure TmePegasusControl.SetSidetoneLevel(const Value: byte);
begin
  if fControlRec.bSidetoneLevel <> Value then
  begin
    fControlRec.bSidetoneLevel := Value;
    SetOnOffByte( fControlRec.bSidetone, fControlRec.bSidetoneLevel, fPegasusProtocol.CWSidetoneVolume );
    DoPropertyChange( wpSidetoneLevel );
  end;
end;

procedure TmePegasusControl.SetSpeechProcessorLevel(const Value: byte);
begin
  if fControlRec.bSpeechProcessorLevel <> Value then
  begin
    fControlRec.bSpeechProcessorLevel := Value;
    SetOnOffByte( fControlRec.bSpeechProcessor, fControlRec.bSpeechProcessorLevel, fPegasusProtocol.SpeechProcessor );
    DoPropertyChange( wpSpeechProcessorLevel );
  end;
end;

procedure TmePegasusControl.SetSpeechProcessor( const Value : Boolean );
begin
  if fControlRec.bSpeechProcessor <> Value then
  begin
    fControlRec.bSpeechProcessor := value;
    SetOnOffByte( fControlRec.bSpeechProcessor, fControlRec.bSpeechProcessorLevel, fPegasusProtocol.SpeechProcessor );
    DoPropertyChange( wpSpeechProcessor );
  end;
end;

procedure TmePegasusControl.SetSpot(const Value: Boolean);
begin
  if fControlRec.bSpot <> value then
  begin
    fControlRec.bSpot := Value;
    SetOnOffByte( fControlRec.bSpot, fControlRec.bSpotLevel, fPegasusProtocol.CWSpotLevel );
    DoPropertyChange( wpSpot );
  end;
end;

procedure TmePegasusControl.SetSpotLevel(const Value: byte);
begin
  if fControlRec.bSpotLevel <> Value then
  begin
    fControlRec.bSpotLevel := Value;
    SetOnOffByte( fControlRec.bSpot, fControlRec.bSpotLevel, fPegasusProtocol.CWSpotLevel );
    DoPropertyChange( wpSpotLevel );
  end;
end;

procedure TmePegasusControl.SetSquelch(const Value: Boolean);
begin
  if fControlRec.bSquelch <> value then
  begin
    fControlRec.bSquelch := Value;
    SetOnOffByte( fControlRec.bSquelch, fControlRec.bSquelchLevel, fPegasusProtocol.Squelch );
    DoPropertyChange( wpSquelch );
  end;
end;

procedure TmePegasusControl.SetSquelchLevel(const Value: byte);
begin
  if fControlRec.bSquelchLevel <> Value then
  begin
    fControlRec.bSquelchLevel := Value;
    SetOnOffByte( fControlRec.bSquelch, fControlRec.bSquelchLevel, fPegasusProtocol.Squelch );
    DoPropertyChange( wpSquelchLevel );
  end;
end;

procedure TmePegasusControl.SetTimeout( const Value: Boolean );
begin
  if fControlRec.bTimeout <> Value then
  begin
    fControlRec.bTimeout := Value;
    fPegasusProtocol.KeepAlive( fControlRec.bTimeout );
    DoPropertyChange( wpTimeout );
  end;
end;

procedure TmePegasusControl.SetTLoop(const Value: Boolean);
begin
  if fControlRec.bTLoop <> value then
  begin
    fControlRec.bTLoop := Value;
    fPegasusProtocol.TLoop( fControlRec.bTLoop );
    DoPropertyChange( wpTLoop );
  end;
end;

procedure TmePegasusControl.SetTxEnable(const Value: Boolean);
begin
  if fControlRec.bTxEnable <> value then
  begin
    fControlRec.bTxEnable := Value;
    fPegasusProtocol.Transmitter( fControlRec.bTxEnable );
    DoPropertyChange( wpTxEnable );
  end;
end;

procedure TmePegasusControl.SetTxFilter(const Value: byte);
begin
  if fVFORec.bTxFilter <> value then
  begin
    fVFORec.bTxFilter := Value;
    if not fInUpdateVFO then
    begin
      fPegasusProtocol.TxFilter( AdjustFilter( fVFORec.bTxFilter ));
      UpdateTxParams;
    end;
    DoPropertyChange( wpTxFilter );
  end;
end;

procedure TmePegasusControl.SetTxFrequency(const Value: integer);
begin
  if fVFORec.bTxFrequency <> value then
  begin
    fVFORec.bTxFrequency := Value;
    if not fInUpdateVFO then
      UpdateTxParams;
    DoPropertyChange( wpTxFrequency );
  end;
end;

procedure TmePegasusControl.SetTxHang(const Value: byte);
begin
  if fControlRec.bTxHang <> Value then
  begin
    fControlRec.bTxHang := Value;
    fPegasusProtocol.TxHang( fControlRec.bTxHang );
    DoPropertyChange( wpTxHang );
  end;
end;

procedure TmePegasusControl.SetTxMode(const Value: byte);
begin
  if fVFORec.bTxMode <> value then
  begin
    fVFORec.bTxMode := Value;
    if not fInUpdateVFO then
    begin
      RxTxModesInt( fVFORec.bRxMode, fVFORec.bTxMode );
      UpdateTxParams;
    end;
    DoPropertyChange( wpTxMode );
  end;
end;

procedure TmePegasusControl.SetTxPBT( const value : Boolean );
begin
  if fVFORec.bTxPBT <> value then
  begin
    fVFORec.bTxPBT := value;
    if not fInUpdateVFO then
      UpdateTxParams;
    DoPropertyChange( wpTxPBT );
  end;
end;

procedure TmePegasusControl.SetTxPBTLevel(const Value: smallint);
begin
  if fVFORec.bTxPBTLevel <> Value then
  begin
    fVFORec.bTxPBTLevel := Value;
    if not fInUpdateVFO then
      UpdateTxParams;
    DoPropertyChange( wpTxPBTLevel );
  end;
end;

procedure TmePegasusControl.SetXIT( const value : Boolean );
begin
  if fVFORec.bXIT <> value then
  begin
    fVFORec.bXIT := value;
    if not fInUpdateVFO then
      UpdateTxParams;
    DoPropertyChange( wpXIT );
  end;
end;

procedure TmePegasusControl.SetXITLevel( const value : smallint );
begin
  if fVFORec.bXITLevel <> value then
  begin
    fVFORec.bXITLevel := value;
    if not fInUpdateVFO then
      UpdateTXParams;
    DoPropertyChange( wpXITLevel );
  end;
end;


procedure TmePegasusControl.SetVOX(const Value: Boolean);
begin
  if fControlRec.bVOX <> Value then
  begin
    fControlRec.bVOX := Value;
    fPegasusProtocol.VOX( fControlRec.bVOX );
    DoPropertyChange( wpVOX );
  end;
end;

procedure TmePegasusControl.SetVOXAnti(const Value: byte);
begin
  if fControlRec.bVOXAnti <> Value then
  begin
    fControlRec.bVOXAnti := Value;
    fPegasusProtocol.VOXAnti( fControlRec.bVOXAnti );
    DoPropertyChange( wpVOXAnti );
  end;
end;

procedure TmePegasusControl.SetVOXGain(const Value: byte);
begin
  if fControlRec.bVOXGain <> Value then
  begin
    fControlRec.bVOXGain := Value;
    fPegasusProtocol.VOXGain( fControlRec.bVOXGain );
    DoPropertyChange( wpVOXGain );
  end;
end;

procedure TmePegasusControl.SetVOXHang(const Value: byte);
begin
  if fControlRec.bVOXHang <> Value then
  begin
    fControlRec.bVOXHang := Value;
    fPegasusProtocol.VOXHang( fControlRec.bVOXHang );
    DoPropertyChange( wpVOXHang );
  end;
end;

procedure TmePegasusControl.SetWidebandTx(const Value: Boolean);
begin
  if fControlRec.bWidebandTx <> Value then
  begin
    fControlRec.bWidebandTx := Value;
    fPegasusProtocol.WideBandTx( fControlRec.bWidebandTx );
    DoPropertyChange( wpWidebandTx );
  end;
end;

procedure TmePegasusControl.SetKeyerSpeed( const value : byte );
begin
  if fControlRec.bKeyerSpeed <> value then
  begin
    fControlRec.bKeyerSpeed := value;
    if not fInUpdateKeyer then
      KeyerParams;
    DoPropertyChange( wpKeyerSpeed );
  end;
end;

procedure TmePegasusControl.SetKeyerWeight( const value : byte );
begin
  if fControlRec.bKeyerWeight <> value then
  begin
    fControlRec.bKeyerWeight := value;
    if not fInUpdateKeyer then
      KeyerParams;
    DoPropertyChange( wpKeyerWeight );
  end;
end;

procedure TmePegasusControl.SetPaddleReverse( const value : Boolean );
begin
  if fControlRec.bPaddleReverse <> value then
  begin
    fControlRec.bPaddleReverse := value;
    if not fInUpdateKeyer then
      KeyerParams;
    DoPropertyChange( wpPaddleReverse );
  end;
end;

procedure TmePegasusControl.SetTunerRFOut( const value : byte );
begin
  if fControlRec.bTunerRFOut <> value then
  begin
    fControlRec.bTunerRFOut := value;
    DoPropertyChange( wpTunerRFOut );
  end;
end;

procedure TmePegasusControl.SetSWRRFOut( const value : byte );
begin
  if fControlRec.bSWRRFOut <> value then
  begin
    fControlRec.bSWRRFOut := value;
    DoPropertyChange( wpSWRRFOut );
  end;
end;

function TmePegasusControl.GetOnChange( const aWhich : TWhichProperty ) : TPegCNotifyEvent;
begin
  result := fOnChangeArray[aWhich]
end;

procedure TmePegasusControl.SetOnChange( const aWhich : TWhichProperty; const aValue : TPegCNotifyEvent );
begin
  fOnChangeArray[aWhich] := aValue;
end;

procedure TmePegasusControl.LoadDefaults;
begin
  fComPort    := DEF_ComPort;
  fVFORec     := DEF_VFORec;
  fControlRec := DEF_ControlRec;
end;


{ --------------------------------------------------------------------------- }
{ ----- tuner support ------------------------------------------------------- }
{ --------------------------------------------------------------------------- }

procedure TmePegasusControl.PT11AutoTune;
begin
  Tuner( PT11_TUNE );
end;

procedure TmePegasusControl.PT11Bypass;
begin
  Tuner( PT11_BYPASS );
end;

procedure TmePegasusControl.PT11CapDown;
begin
  Tuner( PT11_CAPDN );
end;

procedure TmePegasusControl.PT11CapUp;
begin
  Tuner( PT11_CAPUP );
end;

procedure TmePegasusControl.PT11HiZ;
begin
  Tuner( PT11_HIZ );
end;

procedure TmePegasusControl.PT11IndDown;
begin
  Tuner( PT11_INDDN );
end;

procedure TmePegasusControl.PT11IndUp;
begin
  Tuner( PT11_INDUP );
end;

procedure TmePegasusControl.PT11LoZ;
begin
  Tuner( PT11_LOZ );
end;

procedure TmePegasusControl.AT11Tune;
begin
  Tuner( AT11_TUNE );
end;

procedure TmePegasusControl.MeasureSWR;
begin
  Tuner( MEASURE_SWR );
end;

procedure TmePegasusControl.StopTune;
begin
  fTunerStep := tunerNone;
end;

procedure TmePegasusControl.QueryRawAuxStatus;
begin
  fPegasusProtocol.QueryRAWAuxStatus;
end;

procedure TmePegasusControl.QueryForwardPower;
begin
  fPegasusProtocol.QueryForwardPowerReading;
end;

procedure TmePegasusControl.QueryReflectedPower;
begin
  fPegasusProtocol.QueryReflectedPowerReading;
end;

procedure TmePegasusControl.QueryDSPSignalLevel;
begin
  fPegasusProtocol.QueryDSPSignalLevel;
end;

procedure TmePegasusControl.QueryAnalogAGCLevel;
begin
  fPegasusProtocol.QueryAnalogAGCLevel;
end;

procedure Register;
begin
  RegisterComponents('Pegasus', [TmePegasusControl]);
end;

end.

