/* manage the display
 */

#include "GPSReference.h"

#include "version.h"

// call sign to display
#define	CALL	"WB0OEW"

// colors
#define	BG_COLOR	ILI9341_NAVY		// background color
#define	TXT_COLOR	ILI9341_WHITE		// frequency text color
#define	SKY_COLOR	ILI9341_BLUE		// sky background color
#define	SAT_COLOR	ILI9341_CYAN		// satellite marker color
#define	LBL_COLOR	ILI9341_ORANGE		// sky label color

// text sizes
#define	TS_TINY		0
#define	TS_SMALL	1
#define	TS_MEDIUM	2
#define	TS_LARGE	3

// frequency display
#define N_FCH   	13              	// number of frequency characters including punct
#define FCH_W    	(tft_w/N_FCH)		// freq character width
#define FCH_H    	22              	// freq character height @ 3x
#define FR_Y    	20               	// freq text y coord
#define	FUL_Y		(FR_Y+FCH_H+3)		// freq section underline y coord
#define CORR_X    	20        		// correction text x coord
#define CORR_Y    	(FR_Y+FCH_H+10)        	// correction text y coord
#define	OVRHG		4			// overhang

// satellite sky circle
#define SKY_X           (tft_w/2)		// sky center x
#define SKY_Y           (tft_h-tft_w/3)		// sky center y
#define SKY_R           (tft_w/3-2)		// sky radius
#define GPS_R           1			// satellite marker radius
#define SKY_RESTART     (3600000UL)             // refresh interval, milliseconds

// version and call
#define	CALL_X		2			// call x
#define	CALL_Y		(tft_h-22)		// call y
#define	V_X		2			// version x
#define	V_Y		(tft_h-10)		// version y
#define	SNR_W		72			// snr message number width
#define	SNR_H		10			// snr message row height
#define	SNR_X		(tft_w-SNR_W)		// SNR X
#define	SNR_Y		(tft_h-10)		// SNR Y

// power level
#define	PL_X		20			// power level x
#define	PL_Y		(CORR_Y+FCH_H)		// power level y

// time and loc display
#define	TD_X		0			// time and date x
#define	TD_Y		(PL_Y+FCH_H+5)		// time and date y
#define	TD_W		12			// date/time character width
#define	LL_X		12			// lat/long x
#define	LL_Y		(TD_Y+FCH_H)		// lat/long y

// The TFT display uses hardware SPI, plus #9 and #10
#define TFT_DC 9
#define TFT_CS 10

// manage the modulated wave form
static bool mod_active;				// whether modulation is active
static bool mod_on;				// whether modulation is currently on
static uint32_t mod_t0;				// last modulation state change
#define	MOD_MILLIS	250			// modulation duration, ms

/* init the display system
 */
Display::Display ()
{
    // init restart times
    restart_ms = 0;

    // init touch sensor
    ctp = new Adafruit_FT6206();
    if (!ctp->begin(40)) {  // pass in 'sensitivity' coefficient
        Serial.println("Couldn't start FT6206 touchscreen controller");
        while (1);
    }

    // init display
    tft = new Adafruit_ILI9341(TFT_CS, TFT_DC);
    tft->begin();
    tft->fillScreen(BG_COLOR);
    tft_w = tft->width();
    tft_h = tft->height();

    // display version and call
    tft->setTextColor (LBL_COLOR);
    tft->setTextSize(TS_TINY);
    tft->setCursor (V_X, V_Y);
    tft->print('V'); tft->print(VERSION, 2);
    tft->setCursor (CALL_X, CALL_Y);
    tft->print(CALL);

    // set initial control to cHz section
    enc_mult = 1UL;
    tft->drawLine (11*FCH_W, FUL_Y, 13*FCH_W-OVRHG, FUL_Y, TXT_COLOR);
    enc_f0 = dds->getUserFreq();
    Serial.print("init "); Serial.println(enc_f0);

    // display initial user freq
    drawFrequency (enc_f0);

    // display default power level
    drawPower();

    // init prev's
    prev_lat = prev_lng = -999.0;
    prev_bsnr = prev_wsnr = 0; // match gps best/worst_snr
}

/* check for user touching the screen or using the encoder to change frequency
 */
void Display::userFreq()
{
    // update section indication if touched
    if (ctp->touched()) {
	// read and drain touch location
	TS_Point p;
	do {
	    p = ctp->getPoint();
	} while (ctp->touched());
	
	// convert touch location to tft coords
	p.x = map(p.x, 0, tft_w, tft_w, 0);

	// erase current section indicator
	tft->drawLine (0, FUL_Y, tft_w, FUL_Y, BG_COLOR);

	// decide frequency section: update multiplier and section indicator
	if (p.x < 2*FCH_W) {
	    // changing MHz
	    enc_mult = 100000000UL;
	    tft->drawLine (0, FUL_Y, 2*FCH_W-OVRHG, FUL_Y, TXT_COLOR);
	} else if (p.x < 6*FCH_W) {
	    // changing kHz
	    enc_mult = 100000UL;
	    tft->drawLine (3*FCH_W, FUL_Y, 6*FCH_W-OVRHG, FUL_Y, TXT_COLOR);
	} else if (p.x < 10*FCH_W) {
	    // changing Hz
	    enc_mult = 100UL;
	    tft->drawLine (7*FCH_W, FUL_Y, 10*FCH_W-OVRHG, FUL_Y, TXT_COLOR);
	} else {
	    // updating cHz
	    enc_mult = 1UL;
	    tft->drawLine (11*FCH_W, FUL_Y, 13*FCH_W-OVRHG, FUL_Y, TXT_COLOR);
	}

	// set encoder value difference from enc_f0
	enc_f0 = dds->getUserFreq();
	// Serial.print("touch "); Serial.println(enc_f0);
	enc->setValue(0);
    }

    // check encoder activity
    int32_t enc_v;
    if (enc->getValue(enc_v)) {
	if (enc->switchPushed()) {
	    // encoder changed with switch pushed: change power level
	    dds->setPower (dds->getPower() + 1);
	    drawPower();
	} else {
	    // encoder changed with switch up: change frequency
	    (void) dds->setUserFreq(enc_f0 + enc_v*enc_mult);
	    drawFrequency (dds->getUserFreq());
	}
    } else {
	if (enc->switchPushed()) {
	    // switch pushed with no encoder change: toggle modulation
	    if (mod_active = !mod_active)
		mod_t0 = millis();
	    else 
		dds->setOutput (mod_on = true);
	}
    }

    // update modulation if on
    if (mod_active) {
	uint32_t t = millis();
	if (t - mod_t0 > MOD_MILLIS) {
	    dds->setOutput (mod_on = !mod_on);
	    mod_t0 = t;
	}
    }
}

/* draw the given frequency in the format XX,XXX,XXX.XX
 */
void Display::drawFrequency(uint32_t cHz)
{
    tft->fillRect(0, FR_Y, tft_w, FCH_H, BG_COLOR);
    tft->setTextColor (TXT_COLOR);
    tft->setCursor (0, FR_Y);
    tft->setTextSize(TS_LARGE);

    uint32_t frac = 1000000000UL;
    bool any_yet = false;

    // left to right
    for (uint8_t i = 0; i < N_FCH; i++) {
        switch (i) {
        case 2: // FALLTHRU
        case 6:
            tft->print(any_yet ? ',' : ' ');
            break;
        case 10:
	    tft->print('.');
            any_yet = true;
            break;
        default:
            uint8_t d = cHz/frac;
            if (d > 0 || any_yet) {
                tft->print((char)(d + '0'));
                any_yet = true;
            } else
                tft->print(' ');
            cHz -= d*frac;
            frac /= 10U;
            break;
        }
    }
}

/* draw the current output power level
 */
void Display::drawPower()
{
    float p;

    // measured 2017-Dec-23 into 50 ohm load. YMMV
    switch (dds->getPower()) {
    case dds->DDS_2MA: p =  4.7; break;
    case dds->DDS_4MA: p = 10.6; break;
    case dds->DDS_6MA: p = 14.0; break;
    case dds->DDS_8MA: p = 16.6; break;
    default:      p =    0; break;
    }

    tft->fillRect(0, PL_Y, tft_w, FCH_H, BG_COLOR);
    tft->setTextColor (TXT_COLOR);
    tft->setTextSize(TS_MEDIUM);
    tft->setCursor (PL_X, PL_Y);
    tft->print ("Power: ");
    tft->print (p, 1);
    tft->print (" dBm");
}

/* draw the given correction, cHz.
 */
void Display::drawCorrection(int32_t corr)
{
    char buf[32];
    uint8_t l = 0;

    l += sprintf (buf+l, "Error: ");
    if (corr > 9999 || corr < -9999)
	l += sprintf(buf+l, "> 100");
    else {
	dtostrf(fabs(corr)/100.0F, 0, 2, buf+l);	// no %f
	l = strlen(buf);
    }

    l += sprintf (buf+l, " Hz ");
    if (corr > 0)
	l += sprintf(buf+l, "Hi");
    else if (corr < 0)
	l += sprintf(buf+l, "Lo");

    tft->fillRect(0, CORR_Y, tft_w, FCH_H, BG_COLOR);
    tft->setTextColor (TXT_COLOR);
    tft->setTextSize(TS_MEDIUM);
    tft->setCursor (CORR_X, CORR_Y);
    tft->print(buf);
}

/* draw a message in the correction area
 */
void Display::drawMessage (const char *msg)
{
    int16_t x1, y1; uint16_t bw, bh;
    tft->fillRect(0, CORR_Y, tft_w, FCH_H, BG_COLOR);
    tft->setTextColor (TXT_COLOR);
    tft->setTextSize(TS_MEDIUM);
    tft->getTextBounds (msg, 0, 0, &x1, &y1, &bw, &bh);
    tft->setCursor ((tft_w-bw)/2, CORR_Y);
    tft->print(msg);
}


/* draw another satellite
 */
void Display::drawSat(GPS::SatPos &s)
{
    // start over occasionally
    uint32_t ms = millis();
    if (!restart_ms || ms - restart_ms > SKY_RESTART) {
	// fresh sky
	tft->fillCircle(SKY_X, SKY_Y, SKY_R, SKY_COLOR);
	tft->setTextColor (LBL_COLOR);
	tft->setTextSize(TS_MEDIUM);
	tft->setCursor(SKY_X-2, SKY_Y-SKY_R-16);
	tft->print('N');
	tft->setCursor(SKY_X+SKY_R+4, SKY_Y-10);
	tft->print('E');
	tft->setCursor(SKY_X-SKY_R-12, SKY_Y-10);
	tft->print('W');

	// record time
	restart_ms = ms;
    }

    // draw
    uint16_t r = (SKY_R-GPS_R)*(90-s.el)/90;		// [0,90] -> [SKY_R-GPS_R,0]
    float t = (M_PI/180.0F)*(90-s.az);			// radians CW from N
    int16_t dx = r * cos(t);				// pixels right
    int16_t dy = r * sin(t);				// pixels up
    tft->fillCircle(SKY_X+dx,SKY_Y-dy,GPS_R,SAT_COLOR);	// E right N up

    // Serial.print(s.az); Serial.print('\t');
    // Serial.print(s.el); Serial.print('\t');
    // Serial.print(dx); Serial.print('\t');
    // Serial.println(dy);
}

/* draw date, time, and location
 */
void Display::drawNowLoc(GPS::NowLoc &n)
{
    static char months[12][4] = {
	{"Jan"}, {"Feb"}, {"Mar"}, {"Apr"}, {"May"}, {"Jun"},
	{"Jul"}, {"Aug"}, {"Sep"}, {"Oct"}, {"Nov"}, {"Dec"},
    };
    static bool first = true;

    // draw date and time, taking some care to draw minimum necessary to reduce flicker
    tft->setTextColor (LBL_COLOR);
    tft->setTextSize(TS_MEDIUM);

    tft->fillRect(TD_X+18*TD_W, TD_Y, 2*TD_W, 20, BG_COLOR);
    tft->setCursor(TD_X+18*TD_W, TD_Y);
    if (n.second < 10)
	tft->print('0');
    tft->print(n.second);

    if (first || n.second == 0) {
	tft->fillRect(TD_X+15*TD_W, TD_Y, 2*TD_W, 20, BG_COLOR);
	tft->setCursor(TD_X+14*TD_W, TD_Y);
	tft->print(':');
	if (n.minute < 10)
	    tft->print('0');
	tft->print(n.minute);
	tft->print(':');

	if (first || n.minute == 0) {
	    tft->fillRect(TD_X, TD_Y, 14*TD_W, 20, BG_COLOR);
	    tft->setCursor(TD_X, TD_Y);
	    tft->print(n.year+2000);
	    tft->print(' ');
	    tft->print(months[n.month-1]);
	    tft->print(' ');
	    if (n.day < 10)
		tft->print(' ');
	    tft->print(n.day);
	    tft->print(' ');
	    if (n.hour < 10)
		tft->print('0');
	    tft->print(n.hour);

	    first = false;
	}
    }

    // draw lat and long if changed (avoids unnecessary flashing)
    if (n.lat != prev_lat || n.lng != prev_lng) {
	tft->fillRect(0, LL_Y, tft_w, 20, BG_COLOR);
	tft->setTextColor (LBL_COLOR);
	tft->setTextSize(TS_MEDIUM);
	tft->setCursor(LL_X, LL_Y);
	tft->print (n.lng, 4);
	tft->print (n.lng_e ? 'E' : 'W');
	tft->print (' ');
	tft->print (n.lat, 4);
	tft->print (n.lat_n ? 'N' : 'S');

	prev_lat = n.lat;
	prev_lng = n.lng;
    }
}

/* draw SNR range
 */
void Display::drawSNRs (uint8_t worst_snr, uint8_t best_snr)
{
    if (worst_snr != prev_wsnr || best_snr != prev_bsnr) {
	tft->fillRect(SNR_X, SNR_Y, SNR_W, SNR_H, BG_COLOR);
	tft->setTextColor (LBL_COLOR);
	tft->setTextSize(TS_SMALL);
	tft->setCursor(SNR_X, SNR_Y);
	tft->print(F("SNR "));
	tft->print(worst_snr);
	tft->print(F(" .. "));
	tft->print(best_snr);

	prev_wsnr = worst_snr;
	prev_bsnr = best_snr;
    }
}
